''  PCAN-UDS.vb
''
''  ~~~~~~~~~~~~
''
''  PCAN-UDS API
''
''  ~~~~~~~~~~~~
''
''  ------------------------------------------------------------------
''  Author : Fabrice Vergnaud
''	Last changed by:	$Author: Fabrice $
''  Last changed date:	$Date: 2013-11-22 14:02:18 +0100 (ven., 22 nov. 2013) $
''
''  Language: VB.Net
''  ------------------------------------------------------------------
''
''  Copyright (C) 2015  PEAK-System Technik GmbH, Darmstadt
''  more Info at http:''www.peak-system.com
''
Imports System
Imports System.Text
Imports System.Runtime.InteropServices

'' Aliases definition
''
Imports TPUDSCANHandle = System.UInt16     '' Represents a CAN hardware channel of the underlying CAN system

Namespace Peak.Can.Uds


#Region "Enumerations"
    ''' <summary>
    ''' Represents a PCAN Baud rate register value
    ''' </summary>
    Public Enum TPUDSBaudrate As UShort

    ''' <summary>
    ''' 1 MBit/s
    ''' </summary>
        PUDS_BAUD_1M = &H14
        ''' <summary>
        ''' 800 kBit/s
        ''' </summary>
        PUDS_BAUD_800K = &H16
        ''' <summary>
        ''' 500 kBit/s
        ''' </summary>
        PUDS_BAUD_500K = &H1C
        ''' <summary>
        ''' 250 kBit/s
        ''' </summary>
        PUDS_BAUD_250K = &H11C
        ''' <summary>
        ''' 125 kBit/s
        ''' </summary>
        PUDS_BAUD_125K = &H31C
        ''' <summary>
        ''' 100 kBit/s
        ''' </summary>
        PUDS_BAUD_100K = &H432F
        ''' <summary>
        ''' 95,238 kBit/s
        ''' </summary>
        PUDS_BAUD_95K = &HC34E
        ''' <summary>
        ''' 83,333 kBit/s
        ''' </summary>
        PUDS_BAUD_83K = &H852B
        ''' <summary>
        ''' 50 kBit/s
        ''' </summary>
        PUDS_BAUD_50K = &H472F
        ''' <summary>
        ''' 47,619 kBit/s
        ''' </summary>
        PUDS_BAUD_47K = &H1414
        ''' <summary>
        ''' 33,333 kBit/s
        ''' </summary>
        PUDS_BAUD_33K = &H8B2F
        ''' <summary>
        ''' 20 kBit/s
        ''' </summary>
        PUDS_BAUD_20K = &H532F
        ''' <summary>
        ''' 10 kBit/s
        ''' </summary>
        PUDS_BAUD_10K = &H672F
        ''' <summary>
        ''' 5 kBit/s
        ''' </summary>
        PUDS_BAUD_5K = &H7F7F
    End Enum

    ''' <summary>
    ''' Represents the different Not Plug-And-Play PCAN Hardware types
    ''' </summary>
    Public Enum TPUDSHWType As Byte
        ''' <summary>
        ''' PCAN-ISA 82C200
        ''' </summary>
        PUDS_TYPE_ISA = &H1
        ''' <summary>
        ''' PCAN-ISA SJA1000
        ''' </summary>
        PUDS_TYPE_ISA_SJA = &H9
        ''' <summary>
        ''' PHYTEC ISA
        ''' </summary>
        PUDS_TYPE_ISA_PHYTEC = &H4
        ''' <summary>
        ''' PCAN-Dongle 82C200
        ''' </summary>
        PUDS_TYPE_DNG = &H2
        ''' <summary>
        ''' PCAN-Dongle EPP 82C200
        ''' </summary>
        PUDS_TYPE_DNG_EPP = &H3
        ''' <summary>
        ''' PCAN-Dongle SJA1000
        ''' </summary>
        PUDS_TYPE_DNG_SJA = &H5
        ''' <summary>
        ''' PCAN-Dongle EPP SJA1000
        ''' </summary>
        PUDS_TYPE_DNG_SJA_EPP = &H6
    End Enum

    ''' <summary>
    ''' Represent the PUDS error and status codes
    ''' </summary>
    Public Enum TPUDSStatus As UInt32
        ''' <summary>
        ''' No error
        ''' </summary>
        PUDS_ERROR_OK = &H0
        ''' <summary>
        ''' Not Initialized
        ''' </summary>
        PUDS_ERROR_NOT_INITIALIZED = &H1
        ''' <summary>
        ''' Already Initialized
        ''' </summary>
        PUDS_ERROR_ALREADY_INITIALIZED = &H2
        ''' <summary>
        ''' Could not obtain memory
        ''' </summary>
        PUDS_ERROR_NO_MEMORY = &H3
        ''' <summary>
        ''' Input buffer overflow
        ''' </summary>
        PUDS_ERROR_OVERFLOW = &H4
        ''' <summary>
        ''' Timeout while accessing the PCANTP mutex
        ''' </summary>
        PUDS_ERROR_TIMEOUT = &H6
        ''' <summary>
        ''' No Message available
        ''' </summary>
        PUDS_ERROR_NO_MESSAGE = &H7
        ''' <summary>
        ''' Wrong message parameters
        ''' </summary>
        PUDS_ERROR_WRONG_PARAM = &H8
        ''' <summary>
        ''' PCANTP Channel is in BUS-LIGHT error state
        ''' </summary>
        PUDS_ERROR_BUSLIGHT = &H9
        ''' <summary>
        ''' PCANTP Channel is in BUS-HEAVY error state
        ''' </summary>
        PUDS_ERROR_BUSHEAVY = &HA
        ''' <summary>
        ''' PCANTP Channel is in BUS-OFF error state
        ''' </summary>
        PUDS_ERROR_BUSOFF = &HB
        ''' <summary>
        ''' Global CAN error, status code for composition of PCANBasic Errors.
        '''	Remove this value to get a PCAN-Basic TPCANStatus error code.
        ''' </summary>
        PUDS_ERROR_CAN_ERROR = &H80000000UI
    End Enum


    ''' <summary>
    ''' Represents network result values as defined in ISO 15765-2
    ''' </summary>
    Public Enum TPUDSResult As Byte
        ''' <summary>
        ''' general error
        ''' </summary>
        PUDS_RESULT_N_OK = &H0
        ''' <summary>
        ''' timeout occured between 2 frames transmission (sender and receiver side)
        ''' </summary>
        PUDS_RESULT_N_TIMEOUT_A = &H1
        ''' <summary>
        ''' sender side timeout while waiting for flow control frame
        ''' </summary>
        PUDS_RESULT_N_TIMEOUT_BS = &H2
        ''' <summary>
        ''' receiver side timeout while waiting for consecutive frame
        ''' </summary>
        PUDS_RESULT_N_TIMEOUT_CR = &H3
        ''' <summary>
        ''' unexpected sequence number
        ''' </summary>
        PUDS_RESULT_N_WRONG_SN = &H4
        ''' <summary>
        ''' invalid or unknown FlowStatus
        ''' </summary>
        PUDS_RESULT_N_INVALID_FS = &H5
        ''' <summary>
        ''' unexpected protocol data unit
        ''' </summary>
        PUDS_RESULT_N_UNEXP_PDU = &H6
        ''' <summary>
        ''' reception of flow control WAIT frame that exceeds the maximum counter defined by PUDS_PARAM_WFT_MAX
        ''' </summary>
        PUDS_RESULT_N_WFT_OVRN = &H7
        ''' <summary>
        ''' buffer on the receiver side cannot store the data length (server side only)
        ''' </summary>
        PUDS_RESULT_N_BUFFER_OVFLW = &H8
        ''' <summary>
        ''' general error
        ''' </summary>
        PUDS_RESULT_N_ERROR = &H9
    End Enum

    ''' <summary>
    ''' PCANTP parameters
    ''' </summary>
    Public Enum TPUDSParameter As Byte
        ''' <summary>
        ''' 2 BYTE data describing the physical address of the equipment
        ''' </summary>
        PUDS_PARAM_SERVER_ADDRESS = &HC1
        ''' <summary>
        ''' 2 BYTE data (2 BYTE functional address and MSB for status)
        ''' describing a functional address to ignore or listen to
        ''' </summary>
        PUDS_PARAM_SERVER_FILTER = &HC2
        ''' <summary>
        ''' 4 BYTE data describing the maximum time allowed by the client to transmit a request
        ''' See ISO-15765-3 §6.3.2 : /\P2Can_Req
        ''' </summary>
        PUDS_PARAM_TIMEOUT_REQUEST = &HC3
        ''' <summary>
        ''' 4 BYTE data describing the maximum time allowed by the client to receive a response
        ''' See ISO-15765-3 §6.3.2 : /\P2Can_Rsp
        ''' </summary>
        PUDS_PARAM_TIMEOUT_RESPONSE = &HC4
        ''' <summary>
        ''' Require a pointer to a TPUDSSessionInfo structure
        ''' </summary>
        PUDS_PARAM_SESSION_INFO = &HC5
        ''' <summary>
        ''' API version parameter
        ''' </summary>
        PUDS_PARAM_API_VERSION = &HC6
        ''' <summary>
        ''' Define UDS receive-event handler, require a pointer to an event HANDLE.
        ''' </summary>
        PUDS_PARAM_RECEIVE_EVENT = &HC7
        ''' <summary>
        ''' Define a new ISO-TP mapping, requires a pointer to TPUDSMsg containing
        ''' the mapped CAN ID and CAN ID response in the DATA.RAW field.
        ''' </summary>
        PUDS_PARAM_MAPPING_ADD = &HC8
        ''' <summary>
        ''' Remove an ISO-TP mapping, requires a pointer to TPUDSMsg containing the mapped CAN ID to remove.
        ''' </summary>
        PUDS_PARAM_MAPPING_REMOVE = &HC9

        ''' <summary>
        ''' 1 BYTE data describing the block size parameter (BS)
        ''' </summary>
        PUDS_PARAM_BLOCK_SIZE = &HE1
        ''' <summary>
        ''' 1 BYTE data describing the seperation time parameter (STmin)
        ''' </summary>
        PUDS_PARAM_SEPERATION_TIME = &HE2
        ''' <summary>
        ''' 1 BYTE data describing the debug mode
        ''' </summary>
        PUDS_PARAM_DEBUG = &HE3
        ''' <summary>
        ''' 1 Byte data describing the condition of a channel
        ''' </summary>
        PUDS_PARAM_CHANNEL_CONDITION = &HE4
        ''' <summary>
        ''' Integer data describing the Wait Frame Transmissions parameter.
        ''' </summary>
        PUDS_PARAM_WFT_MAX = &HE5
        ''' <summary>
        ''' 1 BYTE data stating if CAN frame DLC uses padding or not
        ''' </summary>
        PUDS_PARAM_CAN_DATA_PADDING = &HE8
        ''' <summary>
        ''' 1 BYTE data stating the value used for CAN data padding
        ''' </summary>
        PUDS_PARAM_PADDING_VALUE = &HED
    End Enum

    ''' <summary>
    ''' PUDS Service IDs defined in ISO 14229-1
    ''' </summary>
    Public Enum TPUDSService As Byte
        PUDS_SI_DiagnosticSessionControl = &H10
        PUDS_SI_ECUReset = &H11
        PUDS_SI_SecurityAccess = &H27
        PUDS_SI_CommunicationControl = &H28
        PUDS_SI_TesterPresent = &H3E
        PUDS_SI_AccessTimingParameter = &H83
        PUDS_SI_SecuredDataTransmission = &H84
        PUDS_SI_ControlDTCSetting = &H85
        PUDS_SI_ResponseOnEvent = &H86
        PUDS_SI_LinkControl = &H87
        PUDS_SI_ReadDataByIdentifier = &H22
        PUDS_SI_ReadMemoryByAddress = &H23
        PUDS_SI_ReadScalingDataByIdentifier = &H24
        PUDS_SI_ReadDataByPeriodicIdentifier = &H2A
        PUDS_SI_DynamicallyDefineDataIdentifier = &H2C
        PUDS_SI_WriteDataByIdentifier = &H2E
        PUDS_SI_WriteMemoryByAddress = &H3D
        PUDS_SI_ClearDiagnosticInformation = &H14
        PUDS_SI_ReadDTCInformation = &H19
        PUDS_SI_InputOutputControlByIdentifier = &H2F
        PUDS_SI_RoutineControl = &H31
        PUDS_SI_RequestDownload = &H34
        PUDS_SI_RequestUpload = &H35
        PUDS_SI_TransferData = &H36
        PUDS_SI_RequestTransferExit = &H37
        ''' <summary>
        ''' Negative response code
        ''' </summary>
        PUDS_NR_SI = &H7F
    End Enum

    ''' <summary>
    ''' PUDS ISO_15765_4 address definitions
    ''' </summary>
    Public Enum TPUDSAddress As Byte
        ''' <summary>
        ''' External test equipment
        ''' </summary>
        PUDS_ISO_15765_4_ADDR_TEST_EQUIPMENT = &HF1
        ''' <summary>
        ''' OBD funtional system
        ''' </summary>
        PUDS_ISO_15765_4_ADDR_OBD_FUNCTIONAL = &H33
        ''' <summary>
        ''' ECU 1
        ''' </summary>
        PUDS_ISO_15765_4_ADDR_ECU_1 = &H1
        ''' <summary>
        ''' ECU 2
        ''' </summary>
        PUDS_ISO_15765_4_ADDR_ECU_2 = &H2
        ''' <summary>
        ''' ECU 3
        ''' </summary>
        PUDS_ISO_15765_4_ADDR_ECU_3 = &H3
        ''' <summary>
        ''' ECU 4
        ''' </summary>
        PUDS_ISO_15765_4_ADDR_ECU_4 = &H4
        ''' <summary>
        ''' ECU 5
        ''' </summary>
        PUDS_ISO_15765_4_ADDR_ECU_5 = &H5
        ''' <summary>
        ''' ECU 6
        ''' </summary>
        PUDS_ISO_15765_4_ADDR_ECU_6 = &H6
        ''' <summary>
        ''' ECU 7
        ''' </summary>
        PUDS_ISO_15765_4_ADDR_ECU_7 = &H7
        ''' <summary>
        ''' ECU 8
        ''' </summary>
        PUDS_ISO_15765_4_ADDR_ECU_8 = &H8
    End Enum

    ''' <summary>
    ''' PUDS ISO_15765_4 11 bit CAN Identifier
    ''' </summary>
    Public Enum TPUDSCanId As UInt32
        ''' <summary>
        ''' CAN ID for functionally addressed request messages sent by external test equipment
        ''' </summary>
        PUDS_ISO_15765_4_CAN_ID_FUNCTIONAL_REQUEST = &H7DF
        ''' <summary>
        ''' physical request CAN ID from external test equipment to ECU #1
        ''' </summary>
        PUDS_ISO_15765_4_CAN_ID_PHYSICAL_REQUEST_1 = &H7E0
        ''' <summary>
        ''' physical response CAN ID from ECU #1 to external test equipment
        ''' </summary>
        PUDS_ISO_15765_4_CAN_ID_PHYSICAL_RESPONSE_1 = &H7E8
        ''' <summary>
        ''' physical request CAN ID from external test equipment to ECU #2
        ''' </summary>
        PUDS_ISO_15765_4_CAN_ID_PHYSICAL_REQUEST_2 = &H7E1
        ''' <summary>
        ''' physical response CAN ID from ECU #2 to external test equipment
        ''' </summary>
        PUDS_ISO_15765_4_CAN_ID_PHYSICAL_RESPONSE_2 = &H7E9
        ''' <summary>
        ''' physical request CAN ID from external test equipment to ECU #3
        ''' </summary>
        PUDS_ISO_15765_4_CAN_ID_PHYSICAL_REQUEST_3 = &H7E2
        ''' <summary>
        ''' physical response CAN ID from ECU #3 to external test equipment
        ''' </summary>
        PUDS_ISO_15765_4_CAN_ID_PHYSICAL_RESPONSE_3 = &H7EA
        ''' <summary>
        ''' physical request CAN ID from external test equipment to ECU #4
        ''' </summary>
        PUDS_ISO_15765_4_CAN_ID_PHYSICAL_REQUEST_4 = &H7E3
        ''' <summary>
        ''' physical response CAN ID from ECU #4 to external test equipment
        ''' </summary>
        PUDS_ISO_15765_4_CAN_ID_PHYSICAL_RESPONSE_4 = &H7EB
        ''' <summary>
        ''' physical request CAN ID from external test equipment to ECU #5
        ''' </summary>
        PUDS_ISO_15765_4_CAN_ID_PHYSICAL_REQUEST_5 = &H7E4
        ''' <summary>
        ''' physical response CAN ID from ECU #5 to external test equipment
        ''' </summary>
        PUDS_ISO_15765_4_CAN_ID_PHYSICAL_RESPONSE_5 = &H7EC
        ''' <summary>
        ''' physical request CAN ID from external test equipment to ECU #6
        ''' </summary>
        PUDS_ISO_15765_4_CAN_ID_PHYSICAL_REQUEST_6 = &H7E5
        ''' <summary>
        ''' physical response CAN ID from ECU #6 to external test equipment
        ''' </summary>
        PUDS_ISO_15765_4_CAN_ID_PHYSICAL_RESPONSE_6 = &H7ED
        ''' <summary>
        ''' physical request CAN ID from external test equipment to ECU #7
        ''' </summary>
        PUDS_ISO_15765_4_CAN_ID_PHYSICAL_REQUEST_7 = &H7E6
        ''' <summary>
        ''' physical response CAN ID from ECU #7 to external test equipment
        ''' </summary>
        PUDS_ISO_15765_4_CAN_ID_PHYSICAL_RESPONSE_7 = &H7EE
        ''' <summary>
        ''' physical request CAN ID from external test equipment to ECU #8
        ''' </summary>
        PUDS_ISO_15765_4_CAN_ID_PHYSICAL_REQUEST_8 = &H7E7
        ''' <summary>
        ''' physical response CAN ID from ECU #8 to external test equipment
        ''' </summary>
        PUDS_ISO_15765_4_CAN_ID_PHYSICAL_RESPONSE_8 = &H7EF
    End Enum

    ''' <summary>
    ''' PUDS Protocol ISO-15765 definitions
    ''' </summary>
    Public Enum TPUDSProtocol As Byte
        ''' <summary>
        ''' non ISO-TP frame (Unacknowledge Unsegmented Data Transfer)
        ''' </summary>
        PUDS_PROTOCOL_NONE = &H0
        ''' <summary>
        ''' using PCAN-ISO-TP with 11 BIT CAN ID, NORMAL addressing and diagnostic message
        ''' </summary>
        PUDS_PROTOCOL_ISO_15765_2_11B = &H1
        ''' <summary>
        ''' using PCAN-ISO-TP with 11 BIT CAN ID, MIXED addressing and remote diagnostic message
        ''' </summary>
        PUDS_PROTOCOL_ISO_15765_2_11B_REMOTE = &H2
        ''' <summary>
        ''' using PCAN-ISO-TP with 29 BIT CAN ID, FIXED NORMAL addressing and diagnostic message
        ''' </summary>
        PUDS_PROTOCOL_ISO_15765_2_29B = &H3
        ''' <summary>
        ''' using PCAN-ISO-TP with 29 BIT CAN ID, MIXED addressing and remote diagnostic message
        ''' </summary>
        PUDS_PROTOCOL_ISO_15765_2_29B_REMOTE = &H4
        ''' <summary>
        ''' using PCAN-ISO-TP with Enhanced diagnostics 29 bit CAN Identifiers
        ''' </summary>
        PUDS_PROTOCOL_ISO_15765_3_29B = &H5
		''' <summary>
		''' using PCAN-ISO-TP with 29 BIT CAN ID, NORMAL addressing and diagnostic message
		''' Note: this protocol requires extra mapping definitions via PCAN-ISO-TP API
		''' </summary>
		PUDS_PROTOCOL_ISO_15765_2_29B_NORMAL = &H6
		''' <summary>
		''' using PCAN-ISO-TP with 11 BIT CAN ID, EXTENDED addressing and diagnostic message
		''' Note: this protocol requires extra mapping definitions via PCAN-ISO-TP API
		''' </summary>
		PUDS_PROTOCOL_ISO_15765_2_11B_EXTENDED = &H7
		''' <summary>
		''' using PCAN-ISO-TP with 29 BIT CAN ID, EXTENDED addressing and diagnostic message
		''' Note: this protocol requires extra mapping definitions via PCAN-ISO-TP API
		''' </summary>
		PUDS_PROTOCOL_ISO_15765_2_29B_EXTENDED = &H8
    End Enum

    ''' <summary>
    ''' PUDS addressing type
    ''' </summary>
    Public Enum TPUDSAddressingType As Byte
        ''' <summary>
        ''' Physical addressing
        ''' </summary>
        PUDS_ADDRESSING_PHYSICAL = &H1
        ''' <summary>
        ''' Functional addressing
        ''' </summary>
        PUDS_ADDRESSING_FUNCTIONAL = &H2
    End Enum

    ''' <summary>
    ''' PCANTP message types
    ''' </summary>
    Public Enum TPUDSMessageType As Byte
        ''' <summary>
        ''' UDS Request Message
        ''' </summary>
        PUDS_MESSAGE_TYPE_REQUEST = &H0
        ''' <summary>
        ''' UDS Request/Response confirmation  Message
        ''' </summary>
        PUDS_MESSAGE_TYPE_CONFIRM = &H1
        ''' <summary>
        ''' Incoming UDS Message
        ''' </summary>
        PUDS_MESSAGE_TYPE_INDICATION = &H2
        ''' <summary>
        ''' UDS Message transmission started
        ''' </summary>
        PUDS_MESSAGE_TYPE_INDICATION_TX = &H3
        ''' <summary>
        ''' Unacknowledge Unsegmented Data Transfert
        ''' </summary>
        PUDS_MESSAGE_TYPE_CONFIRM_UUDT = &H4
    End Enum

    ''' <summary>
    ''' PCANTP Service Result
    ''' </summary>
    Public Enum TPUDSServiceResult As Byte
        ''' <summary>
        ''' Response is valid and matches the requested Service ID.
        ''' </summary>
        Confirmed = &H0
        ''' <summary>
        ''' Response is valid but an Negative Response Code was replied.
        ''' </summary>
        NRC = &H1
        ''' <summary>
        ''' A network error occured in the ISO-TP layer.
        ''' </summary>
        NetworkError = &H2
        ''' <summary>
        ''' Response does not match the requested UDS Service.
        ''' </summary>
        ServiceMismatch = &H3
        ''' <summary>
        ''' Generic error, the message is not a valid response.
        ''' </summary>
        GenericError = &H4
    End Enum
#End Region

#Region "Structures"
    ''' <summary>
    ''' PCAN-UDS Network Addressing Information
    ''' </summary>
    <StructLayout(LayoutKind.Sequential, Pack:=8)>
    Public Structure TPUDSNetAddrInfo
        ''' <summary>
        ''' Represents the origin of this message (address from
        ''' where this message was or will be sent)
        ''' </summary>
        Public SA As Byte
        ''' <summary>
        ''' Represents the destination of this message (address to
        ''' where this message was or will be sent)
        ''' </summary>
        Public TA As Byte
        ''' <summary>
        ''' Represents the kind of addressing being used for communication
        ''' </summary>
        <MarshalAs(UnmanagedType.U1)> _
        Public TA_TYPE As TPUDSAddressingType
        ''' <summary>
        ''' Represents the destination of this message in a remote network
        ''' </summary>
        Public RA As Byte
        ''' <summary>
        ''' Represents the protocol being used for communication
        ''' </summary>
        <MarshalAs(UnmanagedType.U1)> _
        Public PROTOCOL As TPUDSProtocol
    End Structure

    ''' <summary>
    ''' PCAN-UDS Diagnostic Session Information of a server
    ''' </summary>
    <StructLayout(LayoutKind.Sequential, Pack:=8)>
    Public Structure TPUDSSessionInfo
        ''' <summary>
        ''' Network address information
        ''' </summary>
        Public NETADDRINFO As TPUDSNetAddrInfo
        ''' <summary>
        ''' Activated Diagnostic Session (see PUDS_SVC_PARAM_DSC_xxx values)
        ''' </summary>
        Public SESSION_TYPE As Byte
        ''' <summary>
        ''' Default P2Can_Server_Max timing for the activated session
        ''' </summary>
        Public TIMEOUT_P2CAN_SERVER_MAX As UShort
        ''' <summary>
        ''' Enhanced P2Can_Server_Max timing for the activated session
        ''' </summary>
        Public TIMEOUT_ENHANCED_P2CAN_SERVER_MAX As UShort
    End Structure

    ''' <summary>
    ''' PCAN-UDS Message
    ''' </summary>
    <StructLayout(LayoutKind.Sequential, Pack:=8)>
    Public Structure TPUDSMsg
        ''' <summary>
        ''' Network Addressing Information
        ''' </summary>
        Public NETADDRINFO As TPUDSNetAddrInfo
        ''' <summary>
        ''' Result status of the network communication
        ''' </summary>
        <MarshalAs(UnmanagedType.U1)> _
        Public RESULT As TPUDSResult
        ''' <summary>
        ''' States wether Positive Response Message should be suppressed.
        ''' See constants PUDS_SUPPR_POS_RSP_MSG_INDICATION_BIT and PUDS_KEEP_POS_RSP_MSG_INDICATION_BIT.
        ''' </summary>
        Public NO_POSITIVE_RESPONSE_MSG As Byte
        ''' <summary>
        ''' Data Length of the message
        ''' </summary>
        Public LEN As UShort
        ''' <summary>
        ''' Type of UDS Message
        ''' </summary>
        <MarshalAs(UnmanagedType.U1)> _
        Public MSGTYPE As TPUDSMessageType
        ''' <summary>
        ''' Represents the buffer containing the data of this message
        ''' </summary>
        <MarshalAs(UnmanagedType.ByValArray, SizeConst:=4095)> _
        Public DATA As Byte()


        ''' <summary>
        ''' Indicates if this message represents a valid Response
        ''' </summary>
        <Obsolete("IsPositiveResponse is deprecated, please use IsValidResponse instead.")> _
        Public ReadOnly Property IsPositiveResponse() As Boolean
            Get
                Return IsValidResponse
            End Get
        End Property
        ''' <summary>
        ''' Indicates if this message represents a Positive-Response
        ''' </summary>
        Public ReadOnly Property IsValidResponse() As Boolean
            Get
                If (Not IsNothing(DATA)) Then
                    Return (DATA(0) And &H40) = &H40
                End If
                Return False
            End Get
        End Property

        ''' <summary>
        ''' Indicates if this message represents a Negative-Response
        ''' </summary>
        Public ReadOnly Property IsNegativeResponse() As Boolean
            Get
                If (Not IsNothing(DATA)) Then
                    Return DATA(0) = &H7F
                End If
                Return False
            End Get
        End Property

        ''' <summary>
        ''' Shows the data-byte representing the Service-ID of this message
        ''' </summary>
        Public ReadOnly Property ServiceID() As Byte
            Get
                If (Not IsNothing(DATA)) Then
                    If IsNegativeResponse Then
                        Return DATA(1)
                    Else
                        Return DATA(0)
                    End If
                End If
                Return 0
            End Get
        End Property

        ''' <summary>
        ''' Checks if a UDS message is a valid response to a request with the specified Service ID.
        ''' </summary>
        ''' <param name="serviceId">Service ID of the request.</param>
        ''' <param name="nrc">If the response is valid but indicates a UDS error, this parameter will hold the UDS Negative Response Code.</param>
        ''' <returns>Status indicating if the message is confirmed as a positive response.</returns>
        Public Function CheckResponse(ByVal serviceId As TPUDSService, ByRef nrc As Byte) As TPUDSServiceResult
            nrc = 0
            If (RESULT <> TPUDSResult.PUDS_RESULT_N_OK) Then
                Return TPUDSServiceResult.NetworkError
            End If
            If (IsNegativeResponse) Then
                If (DATA(1) <> serviceId) Then
                    Return TPUDSServiceResult.ServiceMismatch
                End If
                nrc = DATA(2)
                Return TPUDSServiceResult.NRC
            ElseIf (IsPositiveResponse) Then
                If (DATA(0) <> (serviceId Or &H40)) Then
                    Return TPUDSServiceResult.ServiceMismatch
                End If
                Return TPUDSServiceResult.Confirmed
            End If
            Return TPUDSServiceResult.GenericError
        End Function
    End Structure
#End Region

#Region "PCAN UDS Api"
    Public NotInheritable Class UDSApi
#Region "PCAN-BUS Handles Definition"
        ''' <summary>
        ''' Undefined/default value for a PCAN bus
        ''' </summary>
        Public Const PUDS_NONEBUS As TPUDSCANHandle = &H0

        ''' <summary>
        ''' PCAN-ISA interface, channel 1
        ''' </summary>
        Public Const PUDS_ISABUS1 As TPUDSCANHandle = &H21
        ''' <summary>
        ''' PCAN-ISA interface, channel 2
        ''' </summary>
        Public Const PUDS_ISABUS2 As TPUDSCANHandle = &H22
        ''' <summary>
        ''' PCAN-ISA interface, channel 3
        ''' </summary>
        Public Const PUDS_ISABUS3 As TPUDSCANHandle = &H23
        ''' <summary>
        ''' PCAN-ISA interface, channel 4
        ''' </summary>
        Public Const PUDS_ISABUS4 As TPUDSCANHandle = &H24
        ''' <summary>
        ''' PCAN-ISA interface, channel 5
        ''' </summary>
        Public Const PUDS_ISABUS5 As TPUDSCANHandle = &H25
        ''' <summary>
        ''' PCAN-ISA interface, channel 6
        ''' </summary>
        Public Const PUDS_ISABUS6 As TPUDSCANHandle = &H26
        ''' <summary>
        ''' PCAN-ISA interface, channel 7
        ''' </summary>
        Public Const PUDS_ISABUS7 As TPUDSCANHandle = &H27
        ''' <summary>
        ''' PCAN-ISA interface, channel 8
        ''' </summary>
        Public Const PUDS_ISABUS8 As TPUDSCANHandle = &H28

        ''' <summary>
        ''' PPCAN-Dongle/LPT interface, channel 1
        ''' </summary>
        Public Const PUDS_DNGBUS1 As TPUDSCANHandle = &H31

        ''' <summary>
        ''' PCAN-PCI interface, channel 1
        ''' </summary>
        Public Const PUDS_PCIBUS1 As TPUDSCANHandle = &H41
        ''' <summary>
        ''' PCAN-PCI interface, channel 2
        ''' </summary>
        Public Const PUDS_PCIBUS2 As TPUDSCANHandle = &H42
        ''' <summary>
        ''' PCAN-PCI interface, channel 3
        ''' </summary>
        Public Const PUDS_PCIBUS3 As TPUDSCANHandle = &H43
        ''' <summary>
        ''' PCAN-PCI interface, channel 4
        ''' </summary>
        Public Const PUDS_PCIBUS4 As TPUDSCANHandle = &H44
        ''' <summary>
        ''' PCAN-PCI interface, channel 5
        ''' </summary>
        Public Const PUDS_PCIBUS5 As TPUDSCANHandle = &H45
        ''' <summary>
        ''' PCAN-PCI interface, channel 6
        ''' </summary>
        Public Const PUDS_PCIBUS6 As TPUDSCANHandle = &H46
        ''' <summary>
        ''' PCAN-PCI interface, channel 7
        ''' </summary>
        Public Const PUDS_PCIBUS7 As TPUDSCANHandle = &H47
        ''' <summary>
        ''' PCAN-PCI interface, channel 8
        ''' </summary>
        Public Const PUDS_PCIBUS8 As TPUDSCANHandle = &H48
        ''' <summary>
        ''' PCAN-PCI interface, channel 9
        ''' </summary>
        Public Const PUDS_PCIBUS9 As TPUDSCANHandle = &H409
        ''' <summary>
        ''' PCAN-PCI interface, channel 10
        ''' </summary>
        Public Const PUDS_PCIBUS10 As TPUDSCANHandle = &H40A
        ''' <summary>
        ''' PCAN-PCI interface, channel 11
        ''' </summary>
        Public Const PUDS_PCIBUS11 As TPUDSCANHandle = &H40B
        ''' <summary>
        ''' PCAN-PCI interface, channel 12
        ''' </summary>
        Public Const PUDS_PCIBUS12 As TPUDSCANHandle = &H40C
        ''' <summary>
        ''' PCAN-PCI interface, channel 13
        ''' </summary>
        Public Const PUDS_PCIBUS13 As TPUDSCANHandle = &H40D
        ''' <summary>
        ''' PCAN-PCI interface, channel 14
        ''' </summary>
        Public Const PUDS_PCIBUS14 As TPUDSCANHandle = &H40E
        ''' <summary>
        ''' PCAN-PCI interface, channel 15
        ''' </summary>
        Public Const PUDS_PCIBUS15 As TPUDSCANHandle = &H40F
        ''' <summary>
        ''' PCAN-PCI interface, channel 16
        ''' </summary>
        Public Const PUDS_PCIBUS16 As TPUDSCANHandle = &H410

        ''' <summary>
        ''' PCAN-USB interface, channel 1
        ''' </summary>
        Public Const PUDS_USBBUS1 As TPUDSCANHandle = &H51
        ''' <summary>
        ''' PCAN-USB interface, channel 2
        ''' </summary>
        Public Const PUDS_USBBUS2 As TPUDSCANHandle = &H52
        ''' <summary>
        ''' PCAN-USB interface, channel 3
        ''' </summary>
        Public Const PUDS_USBBUS3 As TPUDSCANHandle = &H53
        ''' <summary>
        ''' PCAN-USB interface, channel 4
        ''' </summary>
        Public Const PUDS_USBBUS4 As TPUDSCANHandle = &H54
        ''' <summary>
        ''' PCAN-USB interface, channel 5
        ''' </summary>
        Public Const PUDS_USBBUS5 As TPUDSCANHandle = &H55
        ''' <summary>
        ''' PCAN-USB interface, channel 6
        ''' </summary>
        Public Const PUDS_USBBUS6 As TPUDSCANHandle = &H56
        ''' <summary>
        ''' PCAN-USB interface, channel 7
        ''' </summary>
        Public Const PUDS_USBBUS7 As TPUDSCANHandle = &H57
        ''' <summary>
        ''' PCAN-USB interface, channel 8
        ''' </summary>
        Public Const PUDS_USBBUS8 As TPUDSCANHandle = &H58
        ''' <summary>
        ''' PCAN-USB interface, channel 9
        ''' </summary>
        Public Const PUDS_USBBUS9 As TPUDSCANHandle = &H509
        ''' <summary>
        ''' PCAN-USB interface, channel 10
        ''' </summary>
        Public Const PUDS_USBBUS10 As TPUDSCANHandle = &H50A
        ''' <summary>
        ''' PCAN-USB interface, channel 11
        ''' </summary>
        Public Const PUDS_USBBUS11 As TPUDSCANHandle = &H50B
        ''' <summary>
        ''' PCAN-USB interface, channel 12
        ''' </summary>
        Public Const PUDS_USBBUS12 As TPUDSCANHandle = &H50C
        ''' <summary>
        ''' PCAN-USB interface, channel 13
        ''' </summary>
        Public Const PUDS_USBBUS13 As TPUDSCANHandle = &H50D
        ''' <summary>
        ''' PCAN-USB interface, channel 14
        ''' </summary>
        Public Const PUDS_USBBUS14 As TPUDSCANHandle = &H50E
        ''' <summary>
        ''' PCAN-USB interface, channel 15
        ''' </summary>
        Public Const PUDS_USBBUS15 As TPUDSCANHandle = &H50F
        ''' <summary>
        ''' PCAN-USB interface, channel 16
        ''' </summary>
        Public Const PUDS_USBBUS16 As TPUDSCANHandle = &H510

        ''' <summary>
        ''' PCAN-PC Card interface, channel 1
        ''' </summary>
        Public Const PUDS_PCCBUS1 As TPUDSCANHandle = &H61
        ''' <summary>
        ''' PCAN-PC Card interface, channel 2
        ''' </summary>
        Public Const PUDS_PCCBUS2 As TPUDSCANHandle = &H62

        ''' <summary>
        ''' PCAN-LAN interface, channel 1
        ''' </summary>
        Public Const PUDS_LANBUS1 As TPUDSCANHandle = &H801
        ''' <summary>
        ''' PCAN-LAN interface, channel 2
        ''' </summary>
        Public Const PUDS_LANBUS2 As TPUDSCANHandle = &H802
        ''' <summary>
        ''' PCAN-LAN interface, channel 3
        ''' </summary>
        Public Const PUDS_LANBUS3 As TPUDSCANHandle = &H803
        ''' <summary>
        ''' PCAN-LAN interface, channel 4
        ''' </summary>
        Public Const PUDS_LANBUS4 As TPUDSCANHandle = &H804
        ''' <summary>
        ''' PCAN-LAN interface, channel 5
        ''' </summary>
        Public Const PUDS_LANBUS5 As TPUDSCANHandle = &H805
        ''' <summary>
        ''' PCAN-LAN interface, channel 6
        ''' </summary>
        Public Const PUDS_LANBUS6 As TPUDSCANHandle = &H806
        ''' <summary>
        ''' PCAN-LAN interface, channel 7
        ''' </summary>
        Public Const PUDS_LANBUS7 As TPUDSCANHandle = &H807
        ''' <summary>
        ''' PCAN-LAN interface, channel 8
        ''' </summary>
        Public Const PUDS_LANBUS8 As TPUDSCANHandle = &H808
        ''' <summary>
        ''' PCAN-LAN interface, channel 9
        ''' </summary>
        Public Const PUDS_LANBUS9 As TPUDSCANHandle = &H809
        ''' <summary>
        ''' PCAN-LAN interface, channel 10
        ''' </summary>
        Public Const PUDS_LANBUS10 As TPUDSCANHandle = &H80A
        ''' <summary>
        ''' PCAN-LAN interface, channel 11
        ''' </summary>
        Public Const PUDS_LANBUS11 As TPUDSCANHandle = &H80B
        ''' <summary>
        ''' PCAN-LAN interface, channel 12
        ''' </summary>
        Public Const PUDS_LANBUS12 As TPUDSCANHandle = &H80C
        ''' <summary>
        ''' PCAN-LAN interface, channel 13
        ''' </summary>
        Public Const PUDS_LANBUS13 As TPUDSCANHandle = &H80D
        ''' <summary>
        ''' PCAN-LAN interface, channel 14
        ''' </summary>
        Public Const PUDS_LANBUS14 As TPUDSCANHandle = &H80E
        ''' <summary>
        ''' PCAN-LAN interface, channel 15
        ''' </summary>
        Public Const PUDS_LANBUS15 As TPUDSCANHandle = &H80F
        ''' <summary>
        ''' PCAN-LAN interface, channel 16
        ''' </summary>
        Public Const PUDS_LANBUS16 As TPUDSCANHandle = &H810
#End Region

#Region "Parameter values definition"
        ''' <summary>
        ''' No debug messages
        ''' </summary>
        Public Const PUDS_DEBUG_NONE As Byte = 0
        ''' <summary>
        ''' Puts CAN debug messages to stdout
        ''' </summary>
        Public Const PUDS_DEBUG_CAN As Byte = 1
        ''' <summary>
        ''' The Channel is illegal or not available
        ''' </summary>
        Public Const PUDS_CHANNEL_UNAVAILABLE As Byte = 0
        ''' <summary>
        ''' The Channel is available
        ''' </summary>
        Public Const PUDS_CHANNEL_AVAILABLE As Byte = 1
        ''' <summary>
        ''' The Channel is valid, and is being used
        ''' </summary>
        Public Const PUDS_CHANNEL_OCCUPIED As Byte = 2

        ''' <summary>
        ''' Physical address for external test equipment
        ''' </summary>
        Public Const PUDS_SERVER_ADDR_TEST_EQUIPMENT As Byte = &HF1
        ''' <summary>
        ''' Functional request address for Legislated OBD system
        ''' </summary>
        Public Const PUDS_SERVER_ADDR_REQUEST_OBD_SYSTEM As Byte = &H33
        ''' <summary>
        ''' Flag stating that the address is defined as a ISO-15765-3 address
        ''' </summary>
        Public Const PUDS_SERVER_ADDR_FLAG_ENHANCED_ISO_15765_3 As UShort = &H1000
        ''' <summary>
        ''' Mask used for the ISO-15765-3 enhanced addresses
        ''' </summary>
        Public Const PUDS_SERVER_ADDR_MASK_ENHANCED_ISO_15765_3 As UShort = &H7FF
        ''' <summary>
        ''' Filter status : ignore (used to remove previously set filter)
        ''' </summary>
        Public Const PUDS_SERVER_FILTER_IGNORE As UShort = &H0
        ''' <summary>
        ''' Filter status : listen to (must be OR'ed with the 2 BYTE functional address)
        ''' </summary>
        Public Const PUDS_SERVER_FILTER_LISTEN As UShort = &H8000
        ''' <summary>
        ''' Default maximum timeout for UDS transmit confirmation
        ''' </summary>
        Public Const PUDS_TIMEOUT_REQUEST As UInt32 = 10000
        ''' <summary>
        ''' Default maximum timeout for UDS response reception
        ''' </summary>
        Public Const PUDS_TIMEOUT_RESPONSE As UInt32 = 10000
        ''' <summary>
        ''' Default server performance requirement (See ISO-15765-3 §6.3.2)
        ''' </summary>
        Public Const PUDS_P2CAN_DEFAULT_SERVER_MAX As UShort = 50
        ''' <summary>
        ''' Enhanced server performance requirement (See ISO-15765-3 §6.3.2)
        ''' </summary>
        Public Const PUDS_P2CAN_ENHANCED_SERVER_MAX As UShort = 5000
        ''' <summary>
        ''' Uses CAN frame data optimization
        ''' </summary>
        Public Const PUDS_CAN_DATA_PADDING_NONE As UShort = &H0
        ''' <summary>
        ''' Uses CAN frame data padding (default, i.e. CAN DLC = 8)
        ''' </summary>
        Public Const PUDS_CAN_DATA_PADDING_ON As UShort = &H1
        ''' <summary>
        ''' Default value used if CAN data padding is enabled
        ''' </summary>
        Public Const PUDS_CAN_DATA_PADDING_VALUE As UShort = &H55
#End Region

#Region "Values definition related to UDS Message"
        ''' <summary>
        ''' Maximum data length  of UDS messages
        ''' </summary>
        Public Const PUDS_MAX_DATA As UShort = 4095
        ''' <summary>
        ''' Value (for member NO_POSITIVE_RESPONSE_MSG) stating to suppress positive response messages
        ''' </summary>
        Public Const PUDS_SUPPR_POS_RSP_MSG_INDICATION_BIT As Byte = &H80
        ''' <summary>
        ''' Default Value (for member NO_POSITIVE_RESPONSE_MSG) stating to keep positive response messages
        ''' </summary>
        Public Const PUDS_KEEP_POS_RSP_MSG_INDICATION_BIT As Byte = &H0
        ''' <summary>
        ''' Negative response code: Server wants more time
        ''' </summary>
        Public Const PUDS_NRC_EXTENDED_TIMING As Byte = &H78
        ''' <summary>
        ''' Positive response offset
        ''' </summary>
        Public Const PUDS_SI_POSITIVE_RESPONSE As Byte = &H40
#End Region

#Region "PCAN UDS API Implementation"
        ''' <summary>
        ''' Initializes a PUDS-Client based on a PUDS-Channel
        ''' </summary>
        ''' <remarks>Only one UDS-Client can be initialized per CAN-Channel</remarks>
        ''' <param name="CanChannel">The PCAN-Basic channel to be used as UDS client</param>
        ''' <param name="Baudrate">The CAN Hardware speed</param>
        ''' <param name="HwType">NON PLUG'n PLAY: The type of hardware and operation mode</param>
        ''' <param name="IOPort">NON PLUG'n PLAY: The I/O address for the parallel port</param>
        ''' <param name="Interrupt">NON PLUG'n PLAY: Interrupt number of the parallel port</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_Initialize")> _
        Public Shared Function Initialize( _
            <MarshalAs(UnmanagedType.U2)> _
            ByVal CanChannel As TPUDSCANHandle, _
            <MarshalAs(UnmanagedType.U2)> _
            ByVal Baudrate As TPUDSBaudrate, _
            <MarshalAs(UnmanagedType.U1)> _
            ByVal HwType As TPUDSHWType, _
            ByVal IOPort As UInt32, _
            ByVal Interrupt As UInt16) As TPUDSStatus
        End Function


        ''' <summary>
        ''' Initializes a PUDS-Client based on a PUDS-Channel
        ''' </summary>
        ''' <remarks>Only one UDS-Client can be initialized per CAN-Channel</remarks>
        ''' <param name="CanChannel">The PCAN-Basic channel to be used as UDS client</param>
        ''' <param name="Baudrate">The CAN Hardware speed</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        Public Shared Function Initialize( _
            ByVal CanChannel As TPUDSCANHandle, _
            ByVal Baudrate As TPUDSBaudrate) As TPUDSStatus
            Return Initialize(CanChannel, Baudrate, 0, 0, 0)
        End Function

        ''' <summary>
        ''' Uninitializes a PUDS-Client initialized before
        ''' </summary>
        ''' <param name="CanChannel">A PUDS CAN-Channel representing a PUDS-Client</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_Uninitialize")> _
        Public Shared Function Uninitialize( _
            <MarshalAs(UnmanagedType.U2)> _
            ByVal CanChannel As TPUDSCANHandle) As TPUDSStatus
        End Function

        ''' <summary>
        ''' Resets the receive and transmit queues of a PUDS-Client
        ''' </summary>
        ''' <param name="CanChannel">A PUDS CAN-Channel representing a PUDS-Client</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_Reset")> _
        Public Shared Function Reset( _
            <MarshalAs(UnmanagedType.U2)> _
            ByVal CanChannel As TPUDSCANHandle) As TPUDSStatus
        End Function

        ''' <summary>
        ''' Gets information about the internal BUS status of a PUDS CAN-Channel.
        ''' </summary>
        ''' <param name="CanChannel">A PUDS CAN-Channel representing a PUDS-Client</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_GetStatus")> _
        Public Shared Function GetStatus( _
            <MarshalAs(UnmanagedType.U2)> _
            ByVal CanChannel As TPUDSCANHandle) As TPUDSStatus
        End Function

        ''' <summary>
        ''' Reads a PUDS message from the receive queue of a PUDS-Client
        ''' </summary>
        ''' <param name="CanChannel">A PUDS CAN-Channel representing a PUDS-Client</param>
        ''' <param name="MessageBuffer">A TPUDSMsg structure buffer to store the PUDS message</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_Read")> _
        Public Shared Function Read( _
            <MarshalAs(UnmanagedType.U2)> _
            ByVal CanChannel As TPUDSCANHandle, _
            ByRef MessageBuffer As TPUDSMsg) As TPUDSStatus
        End Function

        ''' <summary>
        ''' Transmits a PUDS message
        ''' </summary>
        ''' <param name="CanChannel">A PUDS CAN-Channel representing a PUDS-Client</param>
        ''' <param name="MessageBuffer">A TPUDSMsg buffer with the message to be sent</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_Write")> _
        Public Shared Function Write( _
            <MarshalAs(UnmanagedType.U2)> _
            ByVal CanChannel As TPUDSCANHandle, _
            ByRef MessageBuffer As TPUDSMsg) As TPUDSStatus
        End Function


        ''' <summary>
        ''' Retrieves a PUDS-Client value
        ''' </summary>
        ''' <param name="CanChannel">A PUDS CAN-Channel Handle representing a PUDS-Client</param>
        ''' <param name="Parameter">The TPUDSParameter parameter to get</param>
        ''' <param name="StringBuffer">Buffer for the parameter value</param>
        ''' <param name="BufferLength">Size in bytes of the buffer</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_GetValue")> _
        Public Shared Function GetValue( _
            <MarshalAs(UnmanagedType.U2)> _
            ByVal CanChannel As TPUDSCANHandle, _
            <MarshalAs(UnmanagedType.U1)> _
            ByVal Parameter As TPUDSParameter, _
            ByVal StringBuffer As StringBuilder, _
            ByVal BufferLength As UInt32) As TPUDSStatus
        End Function
        ''' <summary>
        ''' Retrieves a PUDS-Client value
        ''' </summary>
        ''' <param name="CanChannel">A PUDS CAN-Channel Handle representing a PUDS-Client</param>
        ''' <param name="Parameter">The TPUDSParameter parameter to get</param>
        ''' <param name="NumericBuffer">Buffer for the parameter value</param>
        ''' <param name="BufferLength">Size in bytes of the buffer</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_GetValue")> _
        Public Shared Function GetValue( _
            <MarshalAs(UnmanagedType.U2)> _
            ByVal CanChannel As TPUDSCANHandle, _
            <MarshalAs(UnmanagedType.U1)> _
            ByVal Parameter As TPUDSParameter, _
            ByRef NumericBuffer As UInt32, _
            ByVal BufferLength As UInt32) As TPUDSStatus
        End Function
        ''' <summary>
        ''' Retrieves a PUDS-Client value
        ''' </summary>
        ''' <param name="CanChannel">A PUDS CAN-Channel Handle representing a PUDS-Client</param>
        ''' <param name="Parameter">The TPUDSParameter parameter to get</param>
        ''' <param name="Buffer">Buffer for the parameter value</param>
        ''' <param name="BufferLength">Size in bytes of the buffer</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_GetValue")> _
        Public Shared Function GetValue( _
            <MarshalAs(UnmanagedType.U2)> _
            ByVal CanChannel As TPUDSCANHandle, _
            <MarshalAs(UnmanagedType.U1)> _
            ByVal Parameter As TPUDSParameter, _
            <MarshalAs(UnmanagedType.LPArray)> _
            ByVal Buffer As Byte(), _
            ByVal BufferLength As UInt32) As TPUDSStatus
        End Function
        ''' <summary>
        ''' Retrieves a PUDS-Client value
        ''' </summary>
        ''' <param name="CanChannel">A PUDS CAN-Channel Handle representing a PUDS-Client</param>
        ''' <param name="Parameter">The TPUDSParameter parameter to get</param>
        ''' <param name="Buffer">Buffer for the parameter value</param>
        ''' <param name="BufferLength">Size in bytes of the buffer</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_GetValue")> _
        Public Shared Function GetValue( _
            <MarshalAs(UnmanagedType.U2)> _
            ByVal CanChannel As TPUDSCANHandle, _
            <MarshalAs(UnmanagedType.U1)> _
            ByVal Parameter As TPUDSParameter, _
            ByVal Buffer As IntPtr, _
            ByVal BufferLength As UInt32) As TPUDSStatus
        End Function

        ''' <summary>
        ''' Configures or sets a PUDS-Client value
        ''' </summary>
        ''' <param name="CanChannel">A PUDS CAN-Channel Handle representing a PUDS-Client</param>
        ''' <param name="Parameter">The TPUDSParameter parameter to set</param>
        ''' <param name="NumericBuffer">Buffer with the value to be set</param>
        ''' <param name="BufferLength">Size in bytes of the buffer</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_SetValue")> _
        Public Shared Function SetValue( _
            <MarshalAs(UnmanagedType.U2)> _
            ByVal CanChannel As TPUDSCANHandle, _
            <MarshalAs(UnmanagedType.U1)> _
            ByVal Parameter As TPUDSParameter, _
            ByRef NumericBuffer As UInt32, _
            ByVal BufferLength As UInt32) As TPUDSStatus
        End Function
        ''' <summary>
        ''' Configures or sets a PUDS-Client value
        ''' </summary>
        ''' <param name="CanChannel">A PUDS CAN-Channel Handle representing a PUDS-Client</param>
        ''' <param name="Parameter">The TPUDSParameter parameter to set</param>
        ''' <param name="StringBuffer">Buffer with the value to be set</param>
        ''' <param name="BufferLength">Size in bytes of the buffer</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_SetValue")> _
        Public Shared Function SetValue( _
            <MarshalAs(UnmanagedType.U2)> _
            ByVal CanChannel As TPUDSCANHandle, _
            <MarshalAs(UnmanagedType.U1)> _
            ByVal Parameter As TPUDSParameter, _
            <MarshalAs(UnmanagedType.LPStr, SizeParamIndex:=3)> _
            ByVal StringBuffer As String, _
            ByVal BufferLength As UInt32) As TPUDSStatus
        End Function
        ''' <summary>
        ''' Configures or sets a PUDS-Client value
        ''' </summary>
        ''' <param name="CanChannel">A PUDS CAN-Channel Handle representing a PUDS-Client</param>
        ''' <param name="Parameter">The TPUDSParameter parameter to set</param>
        ''' <param name="Buffer">Buffer with the value to be set</param>
        ''' <param name="BufferLength">Size in bytes of the buffer</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_SetValue")> _
        Public Shared Function SetValue( _
            <MarshalAs(UnmanagedType.U2)> _
            ByVal CanChannel As TPUDSCANHandle, _
            <MarshalAs(UnmanagedType.U1)> _
            ByVal Parameter As TPUDSParameter, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=3)> _
            ByVal Buffer As Byte(), _
            ByVal BufferLength As UInt32) As TPUDSStatus
        End Function
        ''' <summary>
        ''' Configures or sets a PUDS-Client value
        ''' </summary>
        ''' <param name="CanChannel">A PUDS CAN-Channel Handle representing a PUDS-Client</param>
        ''' <param name="Parameter">The TPUDSParameter parameter to set</param>
        ''' <param name="Buffer">Buffer with the value to be set</param>
        ''' <param name="BufferLength">Size in bytes of the buffer</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_SetValue")> _
        Public Shared Function SetValue( _
            <MarshalAs(UnmanagedType.U2)> _
            ByVal CanChannel As TPUDSCANHandle, _
            <MarshalAs(UnmanagedType.U1)> _
            ByVal Parameter As TPUDSParameter, _
            ByVal Buffer As IntPtr, _
            ByVal BufferLength As UInt32) As TPUDSStatus
        End Function
#End Region

#Region "PCAN UDS API Implementation : Service handlers"
        ''' <summary>
        ''' Waits for a message (a response or a transmit confirmation) based on a UDS Message Request.
        ''' </summary>
        ''' <param name="CanChannel">A PUDS CAN-Channel representing a PUDS-Client</param>
        ''' <param name="MessageBuffer">A TPUDSMsg structure buffer to store the PUDS response</param>
        ''' <param name="MessageRequest">A sent TPUDSMsg message</param>
        ''' <param name="IsWaitForTransmit">The message to wait for is a Transmit Confirmation or not</param>
        ''' <param name="TimeInterval">Time interval to check for new message</param>
        ''' <param name="Timeout">Maximum time to wait for the message</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_WaitForSingleMessage")> _
        Public Shared Function WaitForSingleMessage( _
            <MarshalAs(UnmanagedType.U2)> _
            ByVal CanChannel As TPUDSCANHandle, _
            ByRef MessageBuffer As TPUDSMsg, _
            ByRef MessageRequest As TPUDSMsg, _
            ByVal IsWaitForTransmit As Boolean, _
            ByVal TimeInterval As UInt32, _
            ByVal Timeout As UInt32) As TPUDSStatus
        End Function

        ''' <summary>
        ''' Waits for multiple messages (multiple responses from a functional request for instance) based on a UDS Message Request.
        ''' </summary>
        ''' <param name="CanChannel">A PUDS CAN-Channel representing a PUDS-Client</param>
        ''' <param name="Buffer">Buffer must be an array of 'MaxCount' entries (must have at least
        ''' a size of iMaxCount * sizeof(TPUDSMsg) bytes</param>
        ''' <param name="MaxCount">Size of the Buffer array (max. messages that can be received)</param>
        ''' <param name="pCount">Buffer for the real number of messages read</param>
        ''' <param name="MessageRequest">A sent TPUDSMsg message</param>
        ''' <param name="TimeInterval">Time interval to check for new message</param>
        ''' <param name="Timeout">Maximum time to wait for the message</param>
        ''' <param name="TimeoutEnhanced">Maximum time to wait for the message in UDS Enhanced mode</param>
        ''' <param name="WaitUntilTimeout">if <code>FALSE</code> the function is interrupted if pCount reaches MaxCount.</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success,
        '''	PUDS_ERROR_OVERFLOW indicates success but Buffer was too small to hold all responses.</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_WaitForMultipleMessage")> _
        Public Shared Function WaitForMultipleMessage( _
            <MarshalAs(UnmanagedType.U2)> _
            ByVal CanChannel As TPUDSCANHandle, _
            <[In](), Out()> ByVal Buffer As TPUDSMsg(), _
            ByVal MaxCount As UInt32, _
            ByRef pCount As UInt32, _
            ByRef MessageRequest As TPUDSMsg, _
            ByVal TimeInterval As UInt32, _
            ByVal Timeout As UInt32, _
            ByVal TimeoutEnhanced As UInt32, _
            ByVal WaitUntilTimeout As Boolean) As TPUDSStatus
        End Function

        ''' <summary>
        ''' Handles the communication workflow for a UDS service expecting a single response.
        ''' </summary>
        '''	<remark>
        '''	The function waits for a transmit confirmation then for a message response.
        '''	Even if the SuppressPositiveResponseMessage flag is set, the function will still wait
        ''' for an eventual Negative Response.
        '''	</remark>
        ''' <param name="CanChannel">A PUDS CAN-Channel representing a PUDS-Client</param>
        ''' <param name="MessageBuffer">A TPUDSMsg structure buffer to store the PUDS response</param>
        ''' <param name="MessageRequest">A sent TPUDSMsg message</param>
        ''' <param name="MessageReqBuffer">A TPUDSMsg structure buffer to store the PUDS request confirmation
        '''	(if <code>NULL</code>, the result confirmation will be set in MessageRequest parameter)</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_WaitForService")> _
        Public Shared Function WaitForService( _
            <MarshalAs(UnmanagedType.U2)> _
            ByVal CanChannel As TPUDSCANHandle, _
            ByRef MessageBuffer As TPUDSMsg, _
            ByRef MessageRequest As TPUDSMsg, _
            ByRef MessageReqBuffer As TPUDSMsg) As TPUDSStatus
        End Function

        ''' <summary>
        ''' Handles the communication workflow for a UDS service expecting multiple responses.
        ''' </summary>
        '''	<remark>
        '''	The function waits for a transmit confirmation then for N message responses.
        '''	Even if the SuppressPositiveResponseMessage flag is set, the function will still wait
        ''' for eventual Negative Responses.
        '''	</remark>
        ''' <param name="CanChannel">A PUDS CAN-Channel representing a PUDS-Client</param>
        ''' <param name="Buffer">Buffer must be an array of 'MaxCount' entries (must have at least
        ''' a size of iMaxCount * sizeof(TPUDSMsg) bytes</param>
        ''' <param name="MaxCount">Size of the Buffer array (max. messages that can be received)</param>
        ''' <param name="pCount">Buffer for the real number of messages read</param>
        ''' <param name="WaitUntilTimeout">if <code>FALSE</code> the function is interrupted if pCount reaches MaxCount.</param>
        ''' <param name="MessageRequest">A sent TPUDSMsg message</param>
        ''' <param name="MessageReqBuffer">A TPUDSMsg structure buffer to store the PUDS request confirmation
        '''	(if <code>NULL</code>, the result confirmation will be set in MessageRequest parameter)</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_WaitForServiceFunctional")> _
        Public Shared Function WaitForServiceFunctional( _
            <MarshalAs(UnmanagedType.U2)> _
            ByVal CanChannel As TPUDSCANHandle, _
            <[In](), Out()> ByVal Buffer As TPUDSMsg(), _
            ByVal MaxCount As UInt32, _
            ByRef pCount As UInt32, _
            ByVal WaitUntilTimeout As Boolean, _
            ByRef MessageRequest As TPUDSMsg, _
            ByRef MessageReqBuffer As TPUDSMsg) As TPUDSStatus
        End Function

        ''' <summary>
        ''' Process a UDS response message to manage ISO-14229/15765 features (like session handling).
        ''' </summary>
        ''' <param name="CanChannel">A PUDS CAN-Channel representing a PUDS-Client</param>
        ''' <param name="MessageBuffer">A TPUDSMsg structure buffer representing the PUDS Response Message</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_ProcessResponse")> _
        Public Shared Function ProcessResponse( _
            <MarshalAs(UnmanagedType.U2)> _
            ByVal CanChannel As TPUDSCANHandle, _
            ByRef MessageBuffer As TPUDSMsg) As TPUDSStatus
        End Function
#End Region

#Region "PCAN UDS API Implementation : Services"

#Region "UDS Service: DiagnosticSessionControl"
        '' ISO-15765-3:2004 §9.2.1 p.42 & ISO-14229-1:2006 §9.2 p.36

        ''' <summary>
        ''' Subfunction parameter for UDS service DiagnosticSessionControl
        ''' </summary>
        Public Enum TPUDSSvcParamDSC As Byte
            ''' <summary>
            ''' Default Session
            ''' </summary>
            PUDS_SVC_PARAM_DSC_DS = &H1
            ''' <summary>
            ''' ECU Programming Session
            ''' </summary>
            PUDS_SVC_PARAM_DSC_ECUPS = &H2
            ''' <summary>
            ''' ECU Extended Diagnostic Session
            ''' </summary>
            PUDS_SVC_PARAM_DSC_ECUEDS = &H3
            ''' <summary>
            ''' Safety System Diagnostic Session
            ''' </summary>
            PUDS_SVC_PARAM_DSC_SSDS = &H4
        End Enum

        ''' <summary>
        ''' The DiagnosticSessionControl service is used to enable different diagnostic sessions in the server.
        ''' </summary>
        ''' <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        ''' <param name="MessageBuffer">The PUDS message</param>
        ''' <param name="SessionType">Subfunction parameter: type of the session (see PUDS_SVC_PARAM_DSC_xxx)</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_SvcDiagnosticSessionControl")> _
        Public Shared Function SvcDiagnosticSessionControl( _
            ByVal CanChannel As TPUDSCANHandle, _
            ByRef MessageBuffer As TPUDSMsg, _
            ByVal SessionType As TPUDSSvcParamDSC) As TPUDSStatus
        End Function

#End Region

#Region "UDS Service: ECUReset"
        '' ISO-15765-3:2004 §9.2.2 p.42 && ISO-14229-1:2006 §9.3 p.42

        ''' <summary>
        ''' Subfunction parameter for UDS service ECURest
        ''' </summary>
        Public Enum TPUDSSvcParamER As Byte
            ''' <summary>
            ''' Hard Reset
            ''' </summary>
            PUDS_SVC_PARAM_ER_HR = &H1
            ''' <summary>
            ''' Key Off on Reset
            ''' </summary>
            PUDS_SVC_PARAM_ER_KOFFONR = &H2
            ''' <summary>
            ''' Soft Reset
            ''' </summary>
            PUDS_SVC_PARAM_ER_SR = &H3
            ''' <summary>
            ''' Enable Rapid Power Shutdown
            ''' </summary>
            PUDS_SVC_PARAM_ER_ERPSD = &H4
            ''' <summary>
            ''' Disable Rapid Power Shutdown
            ''' </summary>
            PUDS_SVC_PARAM_ER_DRPSD = &H5
        End Enum
        ''' <summary>
        ''' The ECUReset service is used by the client to request a server reset.
        ''' </summary>
        ''' <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        ''' <param name="MessageBuffer">The PUDS message</param>
        ''' <param name="ResetType">Subfunction parameter: type of Reset (see PUDS_SVC_PARAM_ER_xxx)</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_SvcECUReset")> _
        Public Shared Function SvcECUReset( _
            ByVal CanChannel As TPUDSCANHandle, _
            ByRef MessageBuffer As TPUDSMsg, _
            ByVal ResetType As TPUDSSvcParamER) As TPUDSStatus
        End Function
#End Region

#Region "UDS Service: SecurityAccess"
        '' ISO-15765-3:2004 §9.2.3 p.43 && ISO-14229-1:2006 §9.4 p.45

        ''' <summary>
        ''' SecurityAccessType : Request Seed and Send Key values
        ''' </summary>
        Public Const PUDS_SVC_PARAM_SA_RSD_1 As Byte = &H1 '' Request Seed
        Public Const PUDS_SVC_PARAM_SA_RSD_3 As Byte = &H3 '' Request Seed
        Public Const PUDS_SVC_PARAM_SA_RSD_5 As Byte = &H5 '' Request Seed
        Public Const PUDS_SVC_PARAM_SA_RSD_MIN As Byte = &H7   '' Request Seed (odd numbers)
        Public Const PUDS_SVC_PARAM_SA_RSD_MAX As Byte = &H5F   '' Request Seed (odd numbers)
        Public Const PUDS_SVC_PARAM_SA_SK_2 As Byte = &H2  '' Send Key
        Public Const PUDS_SVC_PARAM_SA_SK_4 As Byte = &H4  '' Send Key
        Public Const PUDS_SVC_PARAM_SA_SK_6 As Byte = &H6 '' Send Key
        Public Const PUDS_SVC_PARAM_SA_SK_MIN As Byte = &H8    '' Send Key (even numbers)
        Public Const PUDS_SVC_PARAM_SA_SK_MAX As Byte = &H60    '' Send Key (even numbers)

        ''' <summary>
        ''' SecurityAccess service provides a means to access data and/or diagnostic services which have
        '''	restricted access for security, emissions or safety reasons.
        ''' </summary>
        ''' <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        ''' <param name="MessageBuffer">The PUDS message</param>
        ''' <param name="SecurityAccessType">Subfunction parameter: type of SecurityAccess (see PUDS_SVC_PARAM_SA_xxx)</param>
        ''' <param name="Buffer">If Requesting Seed, buffer is the optional data to transmit to a server (like identification).
        '''	If Sending Key, data holds the value generated by the security algorithm corresponding to a specific "seed" value</param>
        ''' <param name="BufferLength">Size in bytes of the buffer</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_SvcSecurityAccess")> _
        Public Shared Function SvcSecurityAccess( _
            ByVal CanChannel As TPUDSCANHandle, _
            ByRef MessageBuffer As TPUDSMsg, _
            ByVal SecurityAccessType As Byte, _
            ByVal Buffer As Byte(), _
            ByVal BufferLength As UShort) As TPUDSStatus
        End Function
#End Region

#Region "UDS Service: CommunicationControl"
        '' ISO-15765-3:2004 §9.2.4 p.43 && ISO-14229-1:2006 §9.5 p.52

        ''' <summary>
        ''' ControlType: Subfunction parameter for UDS service CommunicationControl
        ''' </summary>
        Public Enum TPUDSSvcParamCC As Byte
            ''' <summary>
            ''' Enable Rx and Tx
            ''' </summary>
            PUDS_SVC_PARAM_CC_ERXTX = &H0
            ''' <summary>
            ''' Enable Rx and Disable Tx
            ''' </summary>
            PUDS_SVC_PARAM_CC_ERXDTX = &H1
            ''' <summary>
            ''' Disable Rx and Enable Tx
            ''' </summary>
            PUDS_SVC_PARAM_CC_DRXETX = &H2
            ''' <summary>
            ''' Disable Rx and Tx
            ''' </summary>
            PUDS_SVC_PARAM_CC_DRXTX = &H3
        End Enum

        ''' <summary>
        ''' CommunicationType Flag: Application (01b)
        ''' </summary>
        Public Const PUDS_SVC_PARAM_CC_FLAG_APPL As Byte = &H1
        ''' <summary>
        ''' CommunicationType Flag: NetworkManagement (10b)
        ''' </summary>
        Public Const PUDS_SVC_PARAM_CC_FLAG_NWM As Byte = &H2
        ''' <summary>
        ''' CommunicationType Flag: Disable/Enable specified communicationType (see Flags APPL/NMW)
        ''' </summary>
        Public Const PUDS_SVC_PARAM_CC_FLAG_DESCTIRNCN As Byte = &H0
        '' in the receiving node and all connected networks
        ''' <summary>
        ''' CommunicationType Flag: Disable/Enable network which request is received on
        ''' </summary>
        Public Const PUDS_SVC_PARAM_CC_FLAG_DENWRIRO As Byte = &HF0
        ''' <summary>
        ''' CommunicationType Flag: Disable/Enable specific network identified by network number (minimum value)
        ''' </summary>
        Public Const PUDS_SVC_PARAM_CC_FLAG_DESNIBNN_MIN As Byte = &H10
        ''' <summary>
        ''' CommunicationType Flag: Disable/Enable specific network identified by network number (maximum value)
        ''' </summary>
        Public Const PUDS_SVC_PARAM_CC_FLAG_DESNIBNN_MAX As Byte = &HE0
        ''' <summary>
        ''' CommunicationType Flag: Mask for DESNIBNN bits
        ''' </summary>
        Public Const PUDS_SVC_PARAM_CC_FLAG_DESNIBNN_MASK As Byte = &HF0

        ''' <summary>
        '''	CommunicationControl service's purpose is to switch on/off the transmission
        '''	and/or the reception of certain messages of (a) server(s).
        ''' </summary>
        ''' <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        ''' <param name="MessageBuffer">The PUDS message</param>
        ''' <param name="ControlType">Subfunction parameter: type of CommunicationControl (see PUDS_SVC_PARAM_CC_xxx)</param>
        ''' <param name="CommunicationType">a bit-code value to reference the kind of communication to be controlled,
        '''	See PUDS_SVC_PARAM_CC_FLAG_xxx flags and ISO_14229-2006 §B.1 for bit-encoding</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_SvcCommunicationControl")> _
        Public Shared Function SvcCommunicationControl( _
            ByVal CanChannel As TPUDSCANHandle, _
            ByRef MessageBuffer As TPUDSMsg, _
            ByVal ControlType As TPUDSSvcParamCC, _
            ByVal CommunicationType As Byte) As TPUDSStatus
        End Function
#End Region

#Region "UDS Service: TesterPresent"
        '' ISO-15765-3:2004 §9.2.5 p.43 && ISO-14229-1:2006 §9.6 p.55

        ''' <summary>
        ''' TesterPresentType: Subfunction parameter for UDS service TesterPresent
        ''' </summary>
        Public Enum TPUDSSvcParamTP As Byte
            ''' <summary>
            ''' Zero SubFunction
            ''' </summary>
            PUDS_SVC_PARAM_TP_ZSUBF = &H0
        End Enum

        ''' <summary>
        '''	TesterPresent service indicates to a server (or servers) that a client is still connected
        '''	to the vehicle and that certain diagnostic services and/or communications
        '''	that have been previously activated are to remain active.
        ''' </summary>
        ''' <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        ''' <param name="MessageBuffer">The PUDS message</param>
        ''' <param name="TesterPresentType">No Subfunction parameter by default (PUDS_SVC_PARAM_TP_ZSUBF)</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_SvcTesterPresent")> _
        Public Shared Function SvcTesterPresent( _
            ByVal CanChannel As TPUDSCANHandle, _
            ByRef MessageBuffer As TPUDSMsg, _
            ByVal TesterPresentType As TPUDSSvcParamTP) As TPUDSStatus
        End Function

        ''' <summary>
        '''	TesterPresent service indicates to a server (or servers) that a client is still connected
        '''	to the vehicle and that certain diagnostic services and/or communications
        '''	that have been previously activated are to remain active.
        ''' </summary>
        ''' <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        ''' <param name="MessageBuffer">The PUDS message</param>
        Public Shared Function SvcTesterPresent( _
            ByVal CanChannel As TPUDSCANHandle, _
            ByRef MessageBuffer As TPUDSMsg) As TPUDSStatus
            Return SvcTesterPresent(CanChannel, MessageBuffer, TPUDSSvcParamTP.PUDS_SVC_PARAM_TP_ZSUBF)
        End Function
#End Region

#Region "UDS Service: SecuredDataTransmission"
        '' ISO-15765-3:2004 §9.2.6 p.44 && ISO-14229-1:2006 §9.8 p.63

        ''' <summary>
        '''	SecuredDataTransmission service's purpose is to transmit data that is protected
        '''	against attacks from third parties, which could endanger data security.
        ''' </summary>
        ''' <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        ''' <param name="MessageBuffer">The PUDS message (NO_POSITIVE_RESPONSE_MSG is ignored)</param>
        ''' <param name="Buffer">buffer containing the data as processed by the Security Sub-Layer (See ISO-15764)</param>
        ''' <param name="BufferLength">Size in bytes of the buffer</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_SvcSecuredDataTransmission")> _
        Public Shared Function SvcSecuredDataTransmission( _
            ByVal CanChannel As TPUDSCANHandle, _
            ByRef MessageBuffer As TPUDSMsg, _
            ByVal Buffer As Byte(), _
            ByVal BufferLength As UShort) As TPUDSStatus
        End Function
#End Region

#Region "UDS Service: ControlDTCSetting"
        '' ISO-15765-3:2004 §9.2.7 p.44 && ISO-14229-1:2006 §9.9 p.69

        ''' <summary>
        ''' DTCSettingType: Subfunction parameter for UDS service ControlDTCSetting
        ''' ISO
        ''' </summary>
        Public Enum TPUDSSvcParamCDTCS As Byte
            ''' <summary>
            ''' The server(s) shall resume the setting of diagnostic trouble codes
            ''' </summary>
            PUDS_SVC_PARAM_CDTCS_ON = &H1
            ''' <summary>
            ''' The server(s) shall stop the setting of diagnostic trouble codes
            ''' </summary>
            PUDS_SVC_PARAM_CDTCS_OFF = &H2
        End Enum

        ''' <summary>
        '''	ControlDTCSetting service shall be used by a client to stop or resume the setting of
        '''	diagnostic trouble codes (DTCs) in the server(s).
        ''' </summary>
        ''' <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        ''' <param name="MessageBuffer">The PUDS message</param>
        ''' <param name="DTCSettingType">Subfunction parameter (see PUDS_SVC_PARAM_CDTCS_xxx)</param>
        ''' <param name="Buffer">This parameter record is user-optional and transmits data to a server when controlling the DTC setting.
        '''	It can contain a list of DTCs to be turned on or off.</param>
        ''' <param name="BufferLength">Size in bytes of the buffer</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_SvcControlDTCSetting")> _
        Public Shared Function SvcControlDTCSetting( _
            ByVal CanChannel As TPUDSCANHandle, _
            ByRef MessageBuffer As TPUDSMsg, _
            ByVal DTCSettingType As TPUDSSvcParamCDTCS, _
            ByVal Buffer As Byte(), _
            ByVal BufferLength As UShort) As TPUDSStatus
        End Function
#End Region

#Region "UDS Service: ResponseOnEvent"
        '' ISO-15765-3:2004 §9.2.8 p.44 && ISO-14229-1:2006 §9.10 p.73

        ''' <summary>
        ''' EventType: Subfunction parameter for UDS service ResponseOnEvent
        ''' </summary>
        Public Enum TPUDSSvcParamROE As Byte
            ''' <summary>
            ''' Stop Response On Event
            ''' </summary>
            PUDS_SVC_PARAM_ROE_STPROE = &H0
            ''' <summary>
            ''' On DTC Status Change
            ''' </summary>
            PUDS_SVC_PARAM_ROE_ONDTCS = &H1
            ''' <summary>
            ''' On Timer Interrupt
            ''' </summary>
            PUDS_SVC_PARAM_ROE_OTI = &H2
            ''' <summary>
            ''' On Change Of Data Identifier
            ''' </summary>
            PUDS_SVC_PARAM_ROE_OCODID = &H3
            ''' <summary>
            ''' Report Activated Events
            ''' </summary>
            PUDS_SVC_PARAM_ROE_RAE = &H4
            ''' <summary>
            ''' Start Response On Event
            ''' </summary>
            PUDS_SVC_PARAM_ROE_STRTROE = &H5
            ''' <summary>
            ''' Clear Response On Event
            ''' </summary>
            PUDS_SVC_PARAM_ROE_CLRROE = &H6
            ''' <summary>
            ''' On Comparison Of Values
            ''' </summary>
            PUDS_SVC_PARAM_ROE_OCOV = &H7
        End Enum

        ''' <summary>
        ''' RoE Recommended service (first byte of ServiceToRespondTo Record)
        ''' </summary>
        Public Enum TPUDSSvcParamROERecommendedServiceID As Byte
            PUDS_SVC_PARAM_ROE_STRT_SI_RDBI = TPUDSService.PUDS_SI_ReadDataByIdentifier
            PUDS_SVC_PARAM_ROE_STRT_SI_RDTCI = TPUDSService.PUDS_SI_ReadDTCInformation
            PUDS_SVC_PARAM_ROE_STRT_SI_RC = TPUDSService.PUDS_SI_RoutineControl
            PUDS_SVC_PARAM_ROE_STRT_SI_IOCBI = TPUDSService.PUDS_SI_InputOutputControlByIdentifier
        End Enum

        ''' <summary>
        ''' expected size of EventTypeRecord for ROE_STPROE
        ''' </summary>
        Public Const PUDS_SVC_PARAM_ROE_STPROE_LEN As Byte = 0
        ''' <summary>
        ''' expected size of EventTypeRecord for ROE_ONDTCS
        ''' </summary>
        Public Const PUDS_SVC_PARAM_ROE_ONDTCS_LEN As Byte = 1
        ''' <summary>
        ''' expected size of EventTypeRecord for ROE_OTI
        ''' </summary>
        Public Const PUDS_SVC_PARAM_ROE_OTI_LEN As Byte = 1
        ''' <summary>
        ''' expected size of EventTypeRecord for ROE_OCODID
        ''' </summary>
        Public Const PUDS_SVC_PARAM_ROE_OCODID_LEN As Byte = 2
        ''' <summary>
        ''' expected size of EventTypeRecord for ROE_RAE
        ''' </summary>
        Public Const PUDS_SVC_PARAM_ROE_RAE_LEN As Byte = 0
        ''' <summary>
        ''' expected size of EventTypeRecord for ROE_STRTROE
        ''' </summary>
        Public Const PUDS_SVC_PARAM_ROE_STRTROE_LEN As Byte = 0
        ''' <summary>
        ''' expected size of EventTypeRecord for ROE_CLRROE
        ''' </summary>
        Public Const PUDS_SVC_PARAM_ROE_CLRROE_LEN As Byte = 0
        ''' <summary>
        ''' expected size of EventTypeRecord for ROE_OCOV
        ''' </summary>
        Public Const PUDS_SVC_PARAM_ROE_OCOV_LEN As Byte = 10
        ''' <summary>
        ''' Infinite Time To Response (eventWindowTime parameter)
        ''' </summary>
        Public Const PUDS_SVC_PARAM_ROE_EWT_ITTR As Byte = &H2

        ''' <summary>
        '''	The ResponseOnEvent service requests a server to
        '''	start or stop transmission of responses on a specified event.
        ''' </summary>
        ''' <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        ''' <param name="MessageBuffer">The PUDS message</param>
        ''' <param name="EventType">Subfunction parameter: event type (see PUDS_SVC_PARAM_ROE_xxx)</param>
        ''' <param name="StoreEvent">Storage State (TRUE = Store Event, FALSE = Do Not Store Event)</param>
        ''' <param name="EventWindowTime">Specify a window for the event logic to be active in the server (see PUDS_SVC_PARAM_ROE_EWT_ITTR)</param>
        ''' <param name="EventTypeRecord">Additional parameters for the specified eventType</param>
        ''' <param name="EventTypeRecordLength">Size in bytes of the EventType Record (see PUDS_SVC_PARAM_ROE_xxx_LEN)</param>
        ''' <param name="ServiceToRespondToRecord">Service parameters, with first byte as service Id (see PUDS_SVC_PARAM_ROE_STRT_SI_xxx)</param>
        ''' <param name="ServiceToRespondToRecordLength">Size in bytes of the ServiceToRespondTo Record</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_SvcResponseOnEvent")> _
        Public Shared Function SvcResponseOnEvent( _
            ByVal CanChannel As TPUDSCANHandle, _
            ByRef MessageBuffer As TPUDSMsg, _
            ByVal EventType As TPUDSSvcParamROE, _
            ByVal StoreEvent As Boolean, _
            ByVal EventWindowTime As Byte, _
            ByVal EventTypeRecord As Byte(), _
            ByVal EventTypeRecordLength As UShort, _
            ByVal ServiceToRespondToRecord As Byte(), _
            ByVal ServiceToRespondToRecordLength As UShort) As TPUDSStatus
        End Function
#End Region

#Region "UDS Service: LinkControl"
        '' ISO-15765-3:2004 §9.2.9 p.47 && ISO-14229-1:2006 §9.11 p.91

        ''' <summary>
        ''' LinkControlType: Subfunction parameter for UDS service LinkControl
        ''' </summary>
        Public Enum TPUDSSvcParamLC As Byte
            ''' <summary>
            ''' Verify Baudrate Transition With Fixed Baudrate
            ''' </summary>
            PUDS_SVC_PARAM_LC_VBTWFBR = &H1
            ''' <summary>
            ''' Verify Baudrate Transition With Specific Baudrate
            ''' </summary>
            PUDS_SVC_PARAM_LC_VBTWSBR = &H2
            ''' <summary>
            ''' Transition Baudrate
            ''' </summary>
            PUDS_SVC_PARAM_LC_TB = &H3
        End Enum

        ''' <summary>
        ''' BaudrateIdentifier: standard Baudrate Identifiers
        ''' </summary>
        Public Enum TPUDSSvcParamLCBaudrateIdentifier As Byte
            ''' <summary>
            ''' standard PC baud rate of 9.6 KBaud
            ''' </summary>
            PUDS_SVC_PARAM_LC_BAUDRATE_PC_9600 = &H1
            ''' <summary>
            ''' standard PC baud rate of 19.2 KBaud
            ''' </summary>
            PUDS_SVC_PARAM_LC_BAUDRATE_PC_19200 = &H2
            ''' <summary>
            ''' standard PC baud rate of 38.4 KBaud
            ''' </summary>
            PUDS_SVC_PARAM_LC_BAUDRATE_PC_38400 = &H3
            ''' <summary>
            ''' standard PC baud rate of 57.6 KBaud
            ''' </summary>
            PUDS_SVC_PARAM_LC_BAUDRATE_PC_57600 = &H4
            ''' <summary>
            ''' standard PC baud rate of 115.2 KBaud
            ''' </summary>
            PUDS_SVC_PARAM_LC_BAUDRATE_PC_115200 = &H5
            ''' <summary>
            ''' standard CAN baud rate of 125 KBaud
            ''' </summary>
            PUDS_SVC_PARAM_LC_BAUDRATE_CAN_125K = &H10
            ''' <summary>
            ''' standard CAN baud rate of 250 KBaud
            ''' </summary>
            PUDS_SVC_PARAM_LC_BAUDRATE_CAN_250K = &H11
            ''' <summary>
            ''' standard CAN baud rate of 500 KBaud
            ''' </summary>
            PUDS_SVC_PARAM_LC_BAUDRATE_CAN_500K = &H12
            ''' <summary>
            ''' standard CAN baud rate of 1 MBaud
            ''' </summary>
            PUDS_SVC_PARAM_LC_BAUDRATE_CAN_1M = &H13
        End Enum

        ''' <summary>
        '''	The LinkControl service is used to control the communication link baud rate
        '''	between the client and the server(s) for the exchange of diagnostic data.
        ''' </summary>
        ''' <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        ''' <param name="MessageBuffer">The PUDS message</param>
        ''' <param name="LinkControlType">Subfunction parameter: Link Control Type (see PUDS_SVC_PARAM_LC_xxx)</param>
        ''' <param name="BaudrateIdentifier">defined baud rate identifier (see PUDS_SVC_PARAM_LC_BAUDRATE_xxx)</param>
        ''' <param name="LinkBaudrate">used only with PUDS_SVC_PARAM_LC_VBTWSBR parameter:
        '''	a three-byte value baud rate (baudrate High, Middle and Low Bytes).
        '''	</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success </returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_SvcLinkControl")> _
        Public Shared Function SvcLinkControl( _
            ByVal CanChannel As TPUDSCANHandle, _
            ByRef MessageBuffer As TPUDSMsg, _
            ByVal LinkControlType As TPUDSSvcParamLC, _
            ByVal BaudrateIdentifier As Byte, _
            ByVal LinkBaudrate As UInt32) As TPUDSStatus
        End Function

        ''' <summary>
        '''	The LinkControl service is used to control the communication link baud rate
        '''	between the client and the server(s) for the exchange of diagnostic data.
        ''' </summary>
        ''' <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        ''' <param name="MessageBuffer">The PUDS message</param>
        ''' <param name="LinkControlType">Subfunction parameter: Link Control Type (see PUDS_SVC_PARAM_LC_xxx)</param>
        ''' <param name="BaudrateIdentifier">defined baud rate identifier (see PUDS_SVC_PARAM_LC_BAUDRATE_xxx)</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success </returns>
        Public Shared Function SvcLinkControl( _
            ByVal CanChannel As TPUDSCANHandle, _
            ByRef MessageBuffer As TPUDSMsg, _
            ByVal LinkControlType As TPUDSSvcParamLC, _
            ByVal BaudrateIdentifier As Byte) As TPUDSStatus
            Return SvcLinkControl(CanChannel, MessageBuffer, LinkControlType, BaudrateIdentifier, 0)
        End Function
#End Region

#Region "UDS Service: ReadDataByIdentifier"
        '' ISO-15765-3:2004 §9.3.1 p.47 && ISO-14229-1:2006 §10.2 p.97

        ''' <summary>
        ''' Data Identifiers ISO-14229-1:2006 §C.1 p.259
        ''' </summary>
        Public Enum TPUDSSvcParamDI As UShort
            ''' <summary>
            ''' bootSoftwareIdentificationDataIdentifier
            ''' </summary>
            PUDS_SVC_PARAM_DI_BSIDID = &HF180
            ''' <summary>
            ''' applicationSoftwareIdentificationDataIdentifier
            ''' </summary>
            PUDS_SVC_PARAM_DI_ASIDID = &HF181
            ''' <summary>
            ''' applicationDataIdentificationDataIdentifier
            ''' </summary>
            PUDS_SVC_PARAM_DI_ADIDID = &HF182
            ''' <summary>
            ''' bootSoftwareIdentificationDataIdentifier
            ''' </summary>
            PUDS_SVC_PARAM_DI_BSFPDID = &HF183
            ''' <summary>
            ''' applicationSoftwareFingerprintDataIdentifier
            ''' </summary>
            PUDS_SVC_PARAM_DI_ASFPDID = &HF184
            ''' <summary>
            ''' applicationDataFingerprintDataIdentifier
            ''' </summary>
            PUDS_SVC_PARAM_DI_ADFPDID = &HF185
            ''' <summary>
            ''' activeDiagnosticSessionDataIdentifier
            ''' </summary>
            PUDS_SVC_PARAM_DI_ADSDID = &HF186
            ''' <summary>
            ''' vehicleManufacturerSparePartNumberDataIdentifier
            ''' </summary>
            PUDS_SVC_PARAM_DI_VMSPNDID = &HF187
            ''' <summary>
            ''' vehicleManufacturerECUSoftwareNumberDataIdentifier
            ''' </summary>
            PUDS_SVC_PARAM_DI_VMECUSNDID = &HF188
            ''' <summary>
            ''' vehicleManufacturerECUSoftwareVersionNumberDataIdentifier
            ''' </summary>
            PUDS_SVC_PARAM_DI_VMECUSVNDID = &HF189
            ''' <summary>
            ''' systemSupplierIdentifierDataIdentifier
            ''' </summary>
            PUDS_SVC_PARAM_DI_SSIDDID = &HF18A
            ''' <summary>
            ''' ECUManufacturingDateDataIdentifier
            ''' </summary>
            PUDS_SVC_PARAM_DI_ECUMDDID = &HF18B
            ''' <summary>
            ''' ECUSerialNumberDataIdentifier
            ''' </summary>
            PUDS_SVC_PARAM_DI_ECUSNDID = &HF18C
            ''' <summary>
            ''' supportedFunctionalUnitsDataIdentifier
            ''' </summary>
            PUDS_SVC_PARAM_DI_SFUDID = &HF18D
            ''' <summary>
            ''' vehicleManufacturerKitAssemblyPartNumberDataIdentifier
            ''' </summary>
            PUDS_SVC_PARAM_DI_VMKAPNDID = &HF18E
            ''' <summary>
            ''' VINDataIdentifier
            ''' </summary>
            PUDS_SVC_PARAM_DI_VINDID = &HF190
            ''' <summary>
            ''' vehicleManufacturerECUHardwareNumberDataIdentifier
            ''' </summary>
            PUDS_SVC_PARAM_DI_VMECUHNDID = &HF191
            ''' <summary>
            ''' systemSupplierECUHardwareNumberDataIdentifier
            ''' </summary>
            PUDS_SVC_PARAM_DI_SSECUHWNDID = &HF192
            ''' <summary>
            ''' systemSupplierECUHardwareVersionNumberDataIdentifier
            ''' </summary>
            PUDS_SVC_PARAM_DI_SSECUHWVNDID = &HF193
            ''' <summary>
            ''' systemSupplierECUSoftwareNumberDataIdentifier
            ''' </summary>
            PUDS_SVC_PARAM_DI_SSECUSWNDID = &HF194
            ''' <summary>
            ''' systemSupplierECUSoftwareVersionNumberDataIdentifier
            ''' </summary>
            PUDS_SVC_PARAM_DI_SSECUSWVNDID = &HF195
            ''' <summary>
            ''' exhaustRegulationOrTypeApprovalNumberDataIdentifier
            ''' </summary>
            PUDS_SVC_PARAM_DI_EROTANDID = &HF196
            ''' <summary>
            ''' systemNameOrEngineTypeDataIdentifier
            ''' </summary>
            PUDS_SVC_PARAM_DI_SNOETDID = &HF197
            ''' <summary>
            ''' repairShopCodeOrTesterSerialNumberDataIdentifier
            ''' </summary>
            PUDS_SVC_PARAM_DI_RSCOTSNDID = &HF198
            ''' <summary>
            ''' programmingDateDataIdentifier
            ''' </summary>
            PUDS_SVC_PARAM_DI_PDDID = &HF199
            ''' <summary>
            ''' calibrationRepairShopCodeOrCalibrationEquipmentSerialNumberDataIdentifier
            ''' </summary>
            PUDS_SVC_PARAM_DI_CRSCOCESNDID = &HF19A
            ''' <summary>
            ''' calibrationDateDataIdentifier
            ''' </summary>
            PUDS_SVC_PARAM_DI_CDDID = &HF19B
            ''' <summary>
            ''' calibrationEquipmentSoftwareNumberDataIdentifier
            ''' </summary>
            PUDS_SVC_PARAM_DI_CESWNDID = &HF19C
            ''' <summary>
            ''' ECUInstallationDateDataIdentifier
            ''' </summary>
            PUDS_SVC_PARAM_DI_EIDDID = &HF19D
            ''' <summary>
            ''' ODXFileDataIdentifier
            ''' </summary>
            PUDS_SVC_PARAM_DI_ODXFDID = &HF19E
            ''' <summary>
            ''' entityDataIdentifier
            ''' </summary>
            PUDS_SVC_PARAM_DI_EDID = &HF19F
        End Enum

        ''' <summary>
        '''	The ReadDataByIdentifier service allows the client to request data record values
        '''	from the server identified by one or more dataIdentifiers.
        ''' </summary>
        ''' <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        ''' <param name="MessageBuffer">The PUDS message (NO_POSITIVE_RESPONSE_MSG is ignored)</param>
        ''' <param name="Buffer">buffer containing a list of two-byte Data Identifiers (see PUDS_SVC_PARAM_DI_xxx)</param>
        ''' <param name="BufferLength">Number of elements in the buffer (size in WORD of the buffer)</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_SvcReadDataByIdentifier")> _
        Public Shared Function SvcReadDataByIdentifier( _
            ByVal CanChannel As TPUDSCANHandle, _
            ByRef MessageBuffer As TPUDSMsg, _
            ByVal Buffer As UShort(), _
            ByVal BufferLength As UShort) As TPUDSStatus
        End Function
#End Region

#Region "UDS Service: ReadMemoryByAddress"
        '' ISO-15765-3:2004 §9.3.2 p.47 && ISO-14229-1:2006 §10.3 p.102

        ''' <summary>
        '''	The ReadMemoryByAddress service allows the client to request memory data from the server
        '''	via a provided starting address and to specify the size of memory to be read.
        ''' </summary>
        ''' <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        ''' <param name="MessageBuffer">The PUDS message (NO_POSITIVE_RESPONSE_MSG is ignored)</param>
        ''' <param name="MemoryAddress">starting address of server memory from which data is to be retrieved</param>
        ''' <param name="MemoryAddressLength">Size in bytes of the MemoryAddress buffer (max.: 0xF)</param>
        ''' <param name="MemorySize">number of bytes to be read starting at the address specified by memoryAddress</param>
        ''' <param name="MemorySizeLength">Size in bytes of the MemorySize buffer (max.: 0xF)</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_SvcReadMemoryByAddress")> _
        Public Shared Function SvcReadMemoryByAddress( _
            ByVal CanChannel As TPUDSCANHandle, _
            ByRef MessageBuffer As TPUDSMsg, _
            ByVal MemoryAddress As Byte(), _
            ByVal MemoryAddressLength As Byte, _
            ByVal MemorySize As Byte(), _
            ByVal MemorySizeLength As Byte) As TPUDSStatus
        End Function
#End Region

#Region "UDS Service: ReadScalingDataByIdentifier"
        '' ISO-15765-3:2004 §9.3.3 p.48 && ISO-14229-1:2006 §10.4 p.106

        ''' <summary>
        '''	The ReadScalingDataByIdentifier service allows the client to request
        '''	scaling data record information from the server identified by a dataIdentifier.
        ''' </summary>
        ''' <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        ''' <param name="MessageBuffer">The PUDS message (NO_POSITIVE_RESPONSE_MSG is ignored)</param>
        ''' <param name="DataIdentifier">a two-byte Data Identifier (see PUDS_SVC_PARAM_DI_xxx)</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_SvcReadScalingDataByIdentifier")> _
        Public Shared Function SvcReadScalingDataByIdentifier( _
            ByVal CanChannel As TPUDSCANHandle, _
            ByRef MessageBuffer As TPUDSMsg, _
            ByVal DataIdentifier As UShort) As TPUDSStatus
        End Function
#End Region

#Region "UDS Service: ReadDataByPeriodicIdentifier"
        '' ISO-15765-3:2004 §9.3.4 p.48 && ISO-14229-1:2006 §10.5 p.112

        ''' <summary>
        ''' TransmissionMode: Subfunction parameter for UDS service ReadDataByPeriodicIdentifier
        ''' </summary>
        Public Enum TPUDSSvcParamRDBPI As Byte
            ''' <summary>
            ''' Send At Slow Rate
            ''' </summary>
            PUDS_SVC_PARAM_RDBPI_SASR = &H1
            ''' <summary>
            ''' Send At Medium Rate
            ''' </summary>
            PUDS_SVC_PARAM_RDBPI_SAMR = &H2
            ''' <summary>
            ''' Send At Fast Rate
            ''' </summary>
            PUDS_SVC_PARAM_RDBPI_SAFR = &H3
            ''' <summary>
            ''' Stop Sending
            ''' </summary>
            PUDS_SVC_PARAM_RDBPI_SS = &H4
        End Enum

        ''' <summary>
        '''	The ReadDataByPeriodicIdentifier service allows the client to request the periodic transmission
        '''	of data record values from the server identified by one or more periodicDataIdentifiers.
        ''' </summary>
        ''' <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        ''' <param name="MessageBuffer">The PUDS message (NO_POSITIVE_RESPONSE_MSG is ignored)</param>
        ''' <param name="TransmissionMode">transmission rate code (see PUDS_SVC_PARAM_RDBPI_xxx)</param>
        ''' <param name="Buffer">buffer containing a list of Periodic Data Identifiers</param>
        ''' <param name="BufferLength">Number of elements in the buffer (size in WORD of the buffer)</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_SvcReadDataByPeriodicIdentifier")> _
        Public Shared Function SvcReadDataByPeriodicIdentifier( _
            ByVal CanChannel As TPUDSCANHandle, _
            ByRef MessageBuffer As TPUDSMsg, _
            ByVal TransmissionMode As TPUDSSvcParamRDBPI, _
            ByVal Buffer As Byte(), _
            ByVal BufferLength As UShort) As TPUDSStatus
        End Function
#End Region

#Region "UDS Service: DynamicallyDefineDataIdentifier"
        '' ISO-15765-3:2004 §9.3.5 p.54 && ISO-14229-1:2006 §10.6 p.123

        ''' <summary>
        ''' DynamicallyDefineDataIdentifier Type: Subfunction parameter for UDS service DynamicallyDefineDataIdentifier
        ''' </summary>
        Public Enum TPUDSSvcParamDDDI As Byte
            ''' <summary>
            ''' Define By Identifier
            ''' </summary>
            PUDS_SVC_PARAM_DDDI_DBID = &H1
            ''' <summary>
            ''' Define By Memory Address
            ''' </summary>
            PUDS_SVC_PARAM_DDDI_DBMA = &H2
            ''' <summary>
            ''' Clear Dynamically Defined Data Identifier
            ''' </summary>
            PUDS_SVC_PARAM_DDDI_CDDDI = &H3
        End Enum

        ''' <summary>
        '''	The DynamicallyDefineDataIdentifier service allows the client to dynamically define
        '''	in a server a data identifier that can be read via the ReadDataByIdentifier service at a later time.
        '''	The Define By Identifier subfunction specifies that definition of the dynamic data
        '''	identifier shall occur via a data identifier reference.
        ''' </summary>
        ''' <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        ''' <param name="MessageBuffer">The PUDS message</param>
        ''' <param name="DynamicallyDefinedDataIdentifier">a two-byte Data Identifier (see PUDS_SVC_PARAM_DI_xxx)</param>
        ''' <param name="SourceDataIdentifier">buffer containing the sources of information to be included into the dynamic data record</param>
        ''' <param name="MemorySize">buffer containing the total numbers of bytes from the source data record address</param>
        ''' <param name="PositionInSourceDataRecord">buffer containing the starting byte positions of the excerpt of the source data record</param>
        ''' <param name="BuffersLength">Number of elements in the buffers (SourceDataIdentifier, MemoryAddress and PositionInSourceDataRecord)</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_SvcDynamicallyDefineDataIdentifierDBID")> _
        Public Shared Function SvcDynamicallyDefineDataIdentifierDBID( _
            ByVal CanChannel As TPUDSCANHandle, _
            ByRef MessageBuffer As TPUDSMsg, _
            ByVal DynamicallyDefinedDataIdentifier As UShort, _
            ByVal SourceDataIdentifier As UShort(), _
            ByVal MemorySize As Byte(), _
            ByVal PositionInSourceDataRecord As Byte(), _
            ByVal BuffersLength As UShort) As TPUDSStatus
        End Function

        ''' <summary>
        '''	The DynamicallyDefineDataIdentifier service allows the client to dynamically define
        '''	in a server a data identifier that can be read via the ReadDataByIdentifier service at a later time.
        '''	The Define By Memory Address subfunction specifies that definition of the dynamic data
        '''	identifier shall occur via an address reference.
        ''' </summary>
        ''' <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        ''' <param name="MessageBuffer">The PUDS message</param>
        ''' <param name="DynamicallyDefinedDataIdentifier">a two-byte Data Identifier (see PUDS_SVC_PARAM_DI_xxx)</param>
        ''' <param name="MemoryAddressLength">Size in bytes of the MemoryAddress items in the MemoryAddressBuffer buffer (max.: 0xF)</param>
        ''' <param name="MemorySizeLength">Size in bytes of the MemorySize items in the MemorySizeBuffer buffer (max.: 0xF)</param>
        ''' <param name="MemoryAddressBuffer">buffer containing the MemoryAddress buffers,
        '''	must be an array of 'BuffersLength' entries which contains 'MemoryAddressLength' bytes
        '''	(size is 'BuffersLength * MemoryAddressLength' bytes)</param>
        ''' <param name="MemorySizeBuffer">buffer containing the MemorySize buffers,
        '''	must be an array of 'BuffersLength' entries which contains 'MemorySizeLength' bytes
        '''	(size is 'BuffersLength * MemorySizeLength' bytes)</param>
        ''' <param name="BuffersLength">Size in bytes of the MemoryAddressBuffer and MemorySizeBuffer buffers</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_SvcDynamicallyDefineDataIdentifierDBMA")> _
        Public Shared Function SvcDynamicallyDefineDataIdentifierDBMA( _
            ByVal CanChannel As TPUDSCANHandle, _
            ByRef MessageBuffer As TPUDSMsg, _
            ByVal DynamicallyDefinedDataIdentifier As UShort, _
            ByVal MemoryAddressLength As Byte, _
            ByVal MemorySizeLength As Byte, _
            ByVal MemoryAddressBuffer As Byte(), _
            ByVal MemorySizeBuffer As Byte(), _
            ByVal BuffersLength As UShort) As TPUDSStatus
        End Function

        ''' <summary>
        '''	The Clear Dynamically Defined Data Identifier subfunction shall be used to clear
        '''	the specified dynamic data identifier.
        ''' </summary>
        ''' <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        ''' <param name="MessageBuffer">The PUDS message</param>
        ''' <param name="DynamicallyDefinedDataIdentifier">a two-byte Data Identifier (see PUDS_SVC_PARAM_DI_xxx)</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_SvcDynamicallyDefineDataIdentifierCDDDI")> _
        Public Shared Function SvcDynamicallyDefineDataIdentifierCDDDI( _
            ByVal CanChannel As TPUDSCANHandle, _
            ByRef MessageBuffer As TPUDSMsg, _
            ByVal DynamicallyDefinedDataIdentifier As UShort) As TPUDSStatus
        End Function
#End Region

#Region "UDS Service: WriteDataByIdentifier"
        '' ISO-15765-3:2004 §9.3.6 p.54 && ISO-14229-1:2006 §10.7 p.143

        ''' <summary>
        '''	The WriteDataByIdentifier service allows the client to write information into the server at an internal location
        '''	specified by the provided data identifier.
        ''' </summary>
        ''' <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        ''' <param name="MessageBuffer">The PUDS message (NO_POSITIVE_RESPONSE_MSG is ignored)</param>
        ''' <param name="DataIdentifier">a two-byte Data Identifier (see PUDS_SVC_PARAM_DI_xxx)</param>
        ''' <param name="Buffer">buffer containing the data to write</param>
        ''' <param name="BufferLength">Size in bytes of the buffer</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_SvcWriteDataByIdentifier")> _
        Public Shared Function SvcWriteDataByIdentifier( _
            ByVal CanChannel As TPUDSCANHandle, _
            ByRef MessageBuffer As TPUDSMsg, _
            ByVal DataIdentifier As UShort, _
            ByVal Buffer As Byte(), _
            ByVal BufferLength As UShort) As TPUDSStatus
        End Function
#End Region

#Region "UDS Service: WriteMemoryByAddress"
        '' ISO-15765-3:2004 §9.3.7 p.54 && ISO-14229-1:2006 §10.8 p.146

        ''' <summary>
        '''	The WriteMemoryByAddress service allows the client to write
        '''	information into the server at one or more contiguous memory locations.
        ''' </summary>
        ''' <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        ''' <param name="MessageBuffer">The PUDS message (NO_POSITIVE_RESPONSE_MSG is ignored)</param>
        ''' <param name="MemoryAddress">starting address of server memory to which data is to be written</param>
        ''' <param name="MemoryAddressLength">Size in bytes of the MemoryAddress buffer (max.: 0xF)</param>
        ''' <param name="MemorySize">number of bytes to be written starting at the address specified by memoryAddress</param>
        ''' <param name="MemorySizeLength">Size in bytes of the MemorySize buffer (max.: 0xF)</param>
        ''' <param name="Buffer">buffer containing the data to write</param>
        ''' <param name="BufferLength">Size in bytes of the buffer</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_SvcWriteMemoryByAddress")> _
        Public Shared Function SvcWriteMemoryByAddress( _
            ByVal CanChannel As TPUDSCANHandle, _
            ByRef MessageBuffer As TPUDSMsg, _
            ByVal MemoryAddress As Byte(), _
            ByVal MemoryAddressLength As Byte, _
            ByVal MemorySize As Byte(), _
            ByVal MemorySizeLength As Byte, _
            ByVal Buffer As Byte(), _
            ByVal BufferLength As UShort) As TPUDSStatus
        End Function
#End Region

#Region "UDS Service: ClearDiagnosticInformation"
        '' ISO-15765-3:2004 §9.4.2 p.56 && ISO-14229-1:2006 §11.2 p.152

        ''' <summary>
        ''' groupOfDTC : Emissions-related systems group of DTCs
        ''' </summary>
        Public Const PUDS_SVC_PARAM_CDI_ERS As UInt32 = &H0
        ''' <summary>
        ''' groupOfDTC : All Groups of DTCs
        ''' </summary>
        Public Const PUDS_SVC_PARAM_CDI_AGDTC As UInt32 = &HFFFFFF

        ''' <summary>
        '''	The ClearDiagnosticInformation service is used by the client to clear diagnostic information
        '''	in one server's or multiple servers' memory.
        ''' </summary>
        ''' <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        ''' <param name="MessageBuffer">The PUDS message (NO_POSITIVE_RESPONSE_MSG is ignored)</param>
        ''' <param name="groupOfDTC">a three-byte value indicating the group of DTCs (e.g. powertrain, body, chassis)
        ''' or the particular DTC to be cleared (see PUDS_SVC_PARAM_CDI_xxx)</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_SvcClearDiagnosticInformation")> _
        Public Shared Function SvcClearDiagnosticInformation( _
            ByVal CanChannel As TPUDSCANHandle, _
            ByRef MessageBuffer As TPUDSMsg, _
            ByVal groupOfDTC As UInt32) As TPUDSStatus
        End Function
#End Region

#Region "UDS Service: ReadDTCInformation"
        '' ISO-15765-3:2004 §9.4.1 p.54 && ISO-14229-1:2006 §11.3 p.154

        ''' <summary>
        ''' RDTCIType: Subfunction parameter for UDS service ReadDTCInformation
        ''' ISO-15765-3:2004 §9.4.1 p.54 and ISO-14229-1:2006 §11.3 p.154
        ''' </summary>
        Public Enum TPUDSSvcParamRDTCI As Byte
            ''' <summary>
            ''' report Number Of DTC By Status Mask
            ''' </summary>
            PUDS_SVC_PARAM_RDTCI_RNODTCBSM = &H1
            ''' <summary>
            ''' report DTC By Status Mask
            ''' </summary>
            PUDS_SVC_PARAM_RDTCI_RDTCBSM = &H2
            ''' <summary>
            ''' report DTC Snapshot Identification
            ''' </summary>
            PUDS_SVC_PARAM_RDTCI_RDTCSSI = &H3
            ''' <summary>
            ''' report DTC Snapshot Record By DTC Number
            ''' </summary>
            PUDS_SVC_PARAM_RDTCI_RDTCSSBDTC = &H4
            ''' <summary>
            ''' report DTC Snapshot Record By Record Number
            ''' </summary>
            PUDS_SVC_PARAM_RDTCI_RDTCSSBRN = &H5
            ''' <summary>
            ''' report DTC Extended Data Record By DTC Number
            ''' </summary>
            PUDS_SVC_PARAM_RDTCI_RDTCEDRBDN = &H6
            ''' <summary>
            ''' report Number Of DTC By Severity Mask Record
            ''' </summary>
            PUDS_SVC_PARAM_RDTCI_RNODTCBSMR = &H7
            ''' <summary>
            ''' report DTC By Severity Mask Record
            ''' </summary>
            PUDS_SVC_PARAM_RDTCI_RDTCBSMR = &H8
            ''' <summary>
            ''' report Severity Information Of DTC
            ''' </summary>
            PUDS_SVC_PARAM_RDTCI_RSIODTC = &H9
            ''' <summary>
            ''' report Supported DTC
            ''' </summary>
            PUDS_SVC_PARAM_RDTCI_RSUPDTC = &HA
            ''' <summary>
            ''' report First Test Failed DTC
            ''' </summary>
            PUDS_SVC_PARAM_RDTCI_RFTFDTC = &HB
            ''' <summary>
            ''' report First Confirmed DTC
            ''' </summary>
            PUDS_SVC_PARAM_RDTCI_RFCDTC = &HC
            ''' <summary>
            ''' report Most Recent Test Failed DTC
            ''' </summary>
            PUDS_SVC_PARAM_RDTCI_RMRTFDTC = &HD
            ''' <summary>
            ''' report Most Recent Confirmed DTC
            ''' </summary>
            PUDS_SVC_PARAM_RDTCI_RMRCDTC = &HE
            ''' <summary>
            ''' report Mirror Memory DTC By Status Mask
            ''' </summary>
            PUDS_SVC_PARAM_RDTCI_RMMDTCBSM = &HF
            ''' <summary>
            ''' report Mirror Memory DTC Extended Data Record By DTC Number
            ''' </summary>
            PUDS_SVC_PARAM_RDTCI_RMMDEDRBDN = &H10
            ''' <summary>
            ''' report Number Of Mirror MemoryDTC By Status Mask
            ''' </summary>
            PUDS_SVC_PARAM_RDTCI_RNOMMDTCBSM = &H11
            ''' <summary>
            ''' report Number Of Emissions Related OBD DTC By Status Mask
            ''' </summary>
            PUDS_SVC_PARAM_RDTCI_RNOOBDDTCBSM = &H12
            ''' <summary>
            ''' report Emissions Related OBD DTC By Status Mask
            ''' </summary>
            PUDS_SVC_PARAM_RDTCI_ROBDDTCBSM = &H13
            ''' <summary>
            ''' report DTC Fault Detection Counter
            ''' </summary>
            PUDS_SVC_PARAM_RDTCI_RDTCFDC = &H14
            ''' <summary>
            ''' report DTC With Permanent Status
            ''' </summary>
            PUDS_SVC_PARAM_RDTCI_RDTCWPS = &H15
        End Enum

        ''' <summary>
        ''' DTCSeverityMask (DTCSVM) : ISO-14229-1:2006 §D.3 p.285
        ''' </summary>
        <Flags()> _
        Public Enum TPUDSSvcParamRDTCI_DTCSVM As Byte
            ''' <summary>
            ''' DTC severity bit definitions : no SeverityAvailable
            ''' </summary>
            PUDS_SVC_PARAM_RDTCI_DTCSVM_NSA = &H0
            ''' <summary>
            ''' DTC severity bit definitions : maintenance Only
            ''' </summary>
            PUDS_SVC_PARAM_RDTCI_DTCSVM_MO = &H20
            ''' <summary>
            ''' DTC severity bit definitions : check At Next Halt
            ''' </summary>
            PUDS_SVC_PARAM_RDTCI_DTCSVM_CHKANH = &H40
            ''' <summary>
            ''' DTC severity bit definitions : check Immediately
            ''' </summary>
            PUDS_SVC_PARAM_RDTCI_DTCSVM_CHKI = &H80
        End Enum

        ''' <summary>
        '''	This service allows a client to read the status of server-resident Diagnostic Trouble Code (DTC) information.
        ''' Only reportNumberOfDTCByStatusMask, reportDTCByStatusMask, reportMirrorMemoryDTCByStatusMask,
        '''	reportNumberOfMirrorMemoryDTCByStatusMask, reportNumberOfEmissionsRelatedOBDDTCByStatusMask,
        '''	reportEmissionsRelatedOBDDTCByStatusMask Sub-functions are allowed.
        ''' </summary>
        ''' <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        ''' <param name="MessageBuffer">The PUDS message</param>
        ''' <param name="RDTCIType">Subfunction parameter: ReadDTCInformation type, use one of the following:
        '''	PUDS_SVC_PARAM_RDTCI_RNODTCBSM, PUDS_SVC_PARAM_RDTCI_RDTCBSM,
        '''	PUDS_SVC_PARAM_RDTCI_RMMDTCBSM, PUDS_SVC_PARAM_RDTCI_RNOMMDTCBSM,
        '''	PUDS_SVC_PARAM_RDTCI_RNOOBDDTCBSM, PUDS_SVC_PARAM_RDTCI_ROBDDTCBSM</param>
        ''' <param name="DTCStatusMask">Contains eight DTC status bit.</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_SvcReadDTCInformation")> _
        Public Shared Function SvcReadDTCInformation( _
            ByVal CanChannel As TPUDSCANHandle, _
            ByRef MessageBuffer As TPUDSMsg, _
            ByVal RDTCIType As TPUDSSvcParamRDTCI, _
            ByVal DTCStatusMask As Byte) As TPUDSStatus
        End Function

        ''' <summary>
        '''	This service allows a client to read the status of server-resident Diagnostic Trouble Code (DTC) information.
        '''	The sub-function reportDTCSnapshotRecordByDTCNumber (PUDS_SVC_PARAM_RDTCI_RDTCSSBDTC) is implicit.
        ''' </summary>
        ''' <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        ''' <param name="MessageBuffer">The PUDS message</param>
        ''' <param name="DTCMask">a unique identification number (three byte value) for a specific diagnostic trouble code</param>
        ''' <param name="DTCSnapshotRecordNumber">the number of the specific DTCSnapshot data records</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_SvcReadDTCInformationRDTCSSBDTC")> _
        Public Shared Function SvcReadDTCInformationRDTCSSBDTC( _
            ByVal CanChannel As TPUDSCANHandle, _
            ByRef MessageBuffer As TPUDSMsg, _
            ByVal DTCMask As UInt32, _
            ByVal DTCSnapshotRecordNumber As Byte) As TPUDSStatus
        End Function

        ''' <summary>
        '''	This service allows a client to read the status of server-resident Diagnostic Trouble Code (DTC) information.
        '''	The sub-function reportDTCSnapshotByRecordNumber (PUDS_SVC_PARAM_RDTCI_RDTCSSBRN) is implicit.
        ''' </summary>
        ''' <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        ''' <param name="MessageBuffer">The PUDS message</param>
        ''' <param name="DTCSnapshotRecordNumber">the number of the specific DTCSnapshot data records</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_SvcReadDTCInformationRDTCSSBRN")> _
        Public Shared Function SvcReadDTCInformationRDTCSSBRN( _
            ByVal CanChannel As TPUDSCANHandle, _
            ByRef MessageBuffer As TPUDSMsg, _
            ByVal DTCSnapshotRecordNumber As Byte) As TPUDSStatus
        End Function

        ''' <summary>
        '''	This service allows a client to read the status of server-resident Diagnostic Trouble Code (DTC) information.
        ''' Only reportDTCExtendedDataRecordByDTCNumber and reportMirrorMemoryDTCExtendedDataRecordByDTCNumber Sub-functions are allowed.
        ''' </summary>
        ''' <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        ''' <param name="MessageBuffer">The PUDS message</param>
        ''' <param name="RDTCIType">Subfunction parameter: ReadDTCInformation type, use one of the following:
        '''	PUDS_SVC_PARAM_RDTCI_RDTCEDRBDN, PUDS_SVC_PARAM_RDTCI_RMMDEDRBDN</param>
        ''' <param name="DTCMask">a unique identification number (three byte value) for a specific diagnostic trouble code</param>
        ''' <param name="DTCExtendedDataRecordNumber">the number of the specific DTCExtendedData record requested.</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_SvcReadDTCInformationReportExtended")> _
        Public Shared Function SvcReadDTCInformationReportExtended( _
            ByVal CanChannel As TPUDSCANHandle, _
            ByRef MessageBuffer As TPUDSMsg, _
            ByVal RDTCIType As TPUDSSvcParamRDTCI, _
            ByVal DTCMask As UInt32, _
            ByVal DTCExtendedDataRecordNumber As Byte) As TPUDSStatus
        End Function

        ''' <summary>
        '''	This service allows a client to read the status of server-resident Diagnostic Trouble Code (DTC) information.
        ''' Only reportNumberOfDTCBySeverityMaskRecord and reportDTCSeverityInformation Sub-functions are allowed.
        ''' </summary>
        ''' <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        ''' <param name="MessageBuffer">The PUDS message</param>
        ''' <param name="RDTCIType">Subfunction parameter: ReadDTCInformation type, use one of the following:
        '''	PUDS_SVC_PARAM_RDTCI_RNODTCBSMR, PUDS_SVC_PARAM_RDTCI_RDTCBSMR</param>
        ''' <param name="DTCSeverityMask">a mask of eight (8) DTC severity bits (see PUDS_SVC_PARAM_RDTCI_DTCSVM_xxx)</param>
        ''' <param name="DTCStatusMask">a mask of eight (8) DTC status bits</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_SvcReadDTCInformationReportSeverity")> _
        Public Shared Function SvcReadDTCInformationReportSeverity( _
            ByVal CanChannel As TPUDSCANHandle, _
            ByRef MessageBuffer As TPUDSMsg, _
            ByVal RDTCIType As TPUDSSvcParamRDTCI, _
            ByVal DTCSeverityMask As Byte, _
            ByVal DTCStatusMask As Byte) As TPUDSStatus
        End Function

        ''' <summary>
        '''	This service allows a client to read the status of server-resident Diagnostic Trouble Code (DTC) information.
        '''	The sub-function reportSeverityInformationOfDTC (PUDS_SVC_PARAM_RDTCI_RSIODTC) is implicit.
        ''' </summary>
        ''' <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        ''' <param name="MessageBuffer">The PUDS message</param>
        ''' <param name="DTCMask">a unique identification number for a specific diagnostic trouble code</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_SvcReadDTCInformationRSIODTC")> _
        Public Shared Function SvcReadDTCInformationRSIODTC( _
            ByVal CanChannel As TPUDSCANHandle, _
            ByRef MessageBuffer As TPUDSMsg, _
            ByVal DTCMask As UInt32) As TPUDSStatus
        End Function

        ''' <summary>
        '''	This service allows a client to read the status of server-resident Diagnostic Trouble Code (DTC) information.
        ''' Only reportSupportedDTC, reportFirstTestFailedDTC, reportFirstConfirmedDTC, reportMostRecentTestFailedDTC,
        '''	reportMostRecentConfirmedDTC, reportDTCFaultDetectionCounter, reportDTCWithPermanentStatus,
        ''' and reportDTCSnapshotIdentification Sub-functions are allowed.
        ''' </summary>
        ''' <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        ''' <param name="MessageBuffer">The PUDS message</param>
        ''' <param name="RDTCIType">Subfunction parameter: ReadDTCInformation type, use one of the following:
        '''	PUDS_SVC_PARAM_RDTCI_RFTFDTC, PUDS_SVC_PARAM_RDTCI_RFCDTC,
        '''	PUDS_SVC_PARAM_RDTCI_RMRTFDTC, PUDS_SVC_PARAM_RDTCI_RMRCDTC,
        '''	PUDS_SVC_PARAM_RDTCI_RSUPDTC, PUDS_SVC_PARAM_RDTCI_RDTCWPS,
        '''	PUDS_SVC_PARAM_RDTCI_RDTCSSI</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_SvcReadDTCInformationNoParam")> _
        Public Shared Function SvcReadDTCInformationNoParam( _
            ByVal CanChannel As TPUDSCANHandle, _
            ByRef MessageBuffer As TPUDSMsg, _
            ByVal RDTCIType As TPUDSSvcParamRDTCI) As TPUDSStatus
        End Function
#End Region

#Region "UDS Service: InputOutputControlByIdentifier"
        '' ISO-15765-3:2004 §9.5.1 p.56 && ISO-14229-1:2006 §12.2 p.209,

        ''' <summary>
        ''' inputOutputControlParameter: ISO-14229-1:2006  §E.1 p.289
        ''' </summary>
        Public Enum TPUDSSvcParamIOCBI As Byte
            ''' <summary>
            ''' returnControlToECU (0 controlState bytes in request)
            ''' </summary>
            PUDS_SVC_PARAM_IOCBI_RCTECU = &H0
            ''' <summary>
            ''' resetToDefault (0 controlState bytes in request)
            ''' </summary>
            PUDS_SVC_PARAM_IOCBI_RTD = &H1
            ''' <summary>
            ''' freezeCurrentState (0 controlState bytes in request)
            ''' </summary>
            PUDS_SVC_PARAM_IOCBI_FCS = &H2
            ''' <summary>
            ''' shortTermAdjustment
            ''' </summary>
            PUDS_SVC_PARAM_IOCBI_STA = &H3
        End Enum

        ''' <summary>
        '''	The InputOutputControlByIdentifier service is used by the client to substitute a value for an input signal,
        '''	internal server function and/or control an output (actuator) of an electronic system.
        ''' </summary>
        ''' <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        ''' <param name="MessageBuffer">The PUDS message (NO_POSITIVE_RESPONSE_MSG is ignored)</param>
        ''' <param name="DataIdentifier">a two-byte Data Identifier (see PUDS_SVC_PARAM_DI_xxx)</param>
        ''' <param name="ControlOptionRecord">First byte can be used as either an InputOutputControlParameter
        '''	that describes how the server shall control its inputs or outputs (see PUDS_SVC_PARAM_IOCBI_xxx),
        '''	or as an additional controlState byte</param>
        ''' <param name="ControlOptionRecordLength">Size in bytes of the ControlOptionRecord buffer</param>
        ''' <param name="ControlEnableMaskRecord">The ControlEnableMask shall only be supported when
        '''	the inputOutputControlParameter is used and the dataIdentifier to be controlled consists
        '''	of more than one parameter (i.e. the dataIdentifier is bit-mapped or packeted by definition).
        '''	There shall be one bit in the ControlEnableMask corresponding to each individual parameter
        '''	defined within the dataIdentifier.</param>
        ''' <param name="ControlEnableMaskRecordLength">Size in bytes of the controlEnableMaskRecord buffer</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_SvcInputOutputControlByIdentifier")> _
        Public Shared Function SvcInputOutputControlByIdentifier( _
            ByVal CanChannel As TPUDSCANHandle, _
            ByRef MessageBuffer As TPUDSMsg, _
            ByVal DataIdentifier As UShort, _
            ByVal ControlOptionRecord As Byte(), _
            ByVal ControlOptionRecordLength As UShort, _
            ByVal ControlEnableMaskRecord As Byte(), _
            ByVal ControlEnableMaskRecordLength As UShort) As TPUDSStatus
        End Function
#End Region

#Region "UDS Service: RoutineControl"
        '' ISO-15765-3:2004 §9.6.1 p.56 && ISO-14229-1:2006 §13.2 p.225

        ''' <summary>
        ''' RoutineControlType: Subfunction parameter for UDS service RoutineControl
        ''' </summary>
        Public Enum TPUDSSvcParamRC As Byte
            ''' <summary>
            ''' Start Routine
            ''' </summary>
            PUDS_SVC_PARAM_RC_STR = &H1
            ''' <summary>
            ''' Stop Routine
            ''' </summary>
            PUDS_SVC_PARAM_RC_STPR = &H2
            ''' <summary>
            ''' Request Routine Results
            ''' </summary>
            PUDS_SVC_PARAM_RC_RRR = &H3
        End Enum

        ''' <summary>
        ''' Routine Identifier: ISO-14229-1:2006 §F.1 p.290
        ''' </summary>
        Public Enum TPUDSSvcParamRC_RID As UShort
            ''' <summary>
            ''' Deploy Loop Routine ID
            ''' </summary>
            PUDS_SVC_PARAM_RC_RID_DLRI_ = &HE200
            ''' <summary>
            ''' erase Memory
            ''' </summary>
            PUDS_SVC_PARAM_RC_RID_EM_ = &HFF00
            ''' <summary>
            ''' check Programming Dependencies
            ''' </summary>
            PUDS_SVC_PARAM_RC_RID_CPD_ = &HFF01
            ''' <summary>
            ''' erase Mirror Memory DTCs
            ''' </summary>
            PUDS_SVC_PARAM_RC_RID_EMMDTC_ = &HFF02
        End Enum

        ''' <summary>
        '''	The RoutineControl service is used by the client to start/stop a routine,
        '''	and request routine results.
        ''' </summary>
        ''' <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        ''' <param name="MessageBuffer">The PUDS message</param>
        ''' <param name="RoutineControlType">Subfunction parameter: RoutineControl type (see PUDS_SVC_PARAM_RC_xxx)</param>
        ''' <param name="RoutineIdentifier">Server Local Routine Identifier (see PUDS_SVC_PARAM_RC_RID_xxx)</param>
        ''' <param name="Buffer">buffer containing the Routine Control Options (only with start and stop routine sub-functions)</param>
        ''' <param name="BufferLength">Size in bytes of the buffer</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_SvcRoutineControl")> _
        Public Shared Function SvcRoutineControl( _
            ByVal CanChannel As TPUDSCANHandle, _
            ByRef MessageBuffer As TPUDSMsg, _
            ByVal RoutineControlType As TPUDSSvcParamRC, _
            ByVal RoutineIdentifier As UShort, _
            ByVal Buffer As Byte(), _
            ByVal BufferLength As UShort) As TPUDSStatus
        End Function
#End Region

#Region "UDS Service: requestDownload"
        '' ISO-15765-3:2004 §9.7.1 p.57 && ISO-14229-1:2006 §14.2 p.231

        ''' <summary>
        '''	The requestDownload service is used by the client to initiate a data transfer
        '''	from the client to the server (download).
        ''' </summary>
        ''' <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        ''' <param name="MessageBuffer">The PUDS message (NO_POSITIVE_RESPONSE_MSG is ignored)</param>
        ''' <param name="CompressionMethod">A nibble-value that specifies the "compressionMethod",
        '''	The value 0x0 specifies that no compressionMethod is used.</param>
        ''' <param name="EncryptingMethod">A nibble-value that specifies the "encryptingMethod",
        '''	The value 0x0 specifies that no encryptingMethod is used.</param>
        ''' <param name="MemoryAddress">starting address of server memory to which data is to be written</param>
        ''' <param name="MemoryAddressLength">Size in bytes of the MemoryAddress buffer (max.: 0xF)</param>
        ''' <param name="MemorySize">used by the server to compare the uncompressed memory size with
        '''	the total amount of data transferred during the TransferData service</param>
        ''' <param name="MemorySizeLength">Size in bytes of the MemorySize buffer (max.: 0xF)</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_SvcRequestDownload")> _
        Public Shared Function SvcRequestDownload( _
            ByVal CanChannel As TPUDSCANHandle, _
            ByRef MessageBuffer As TPUDSMsg, _
            ByVal CompressionMethod As Byte, _
            ByVal EncryptingMethod As Byte, _
            ByVal MemoryAddress As Byte(), _
            ByVal MemoryAddressLength As Byte, _
            ByVal MemorySize As Byte(), _
            ByVal MemorySizeLength As Byte) As TPUDSStatus
        End Function
#End Region

#Region "UDS Service: requestUpload"
        '' ISO-15765-3:2004 §9.7.1 p.57 && ISO-14229-1:2006 §14.3 p.234

        ''' <summary>
        '''	The requestUpload service is used by the client to initiate a data transfer
        '''	from the server to the client (upload).
        ''' </summary>
        ''' <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        ''' <param name="MessageBuffer">The PUDS message (NO_POSITIVE_RESPONSE_MSG is ignored)</param>
        ''' <param name="CompressionMethod">A nibble-value that specifies the "compressionMethod",
        '''	The value 0x0 specifies that no compressionMethod is used.</param>
        ''' <param name="EncryptingMethod">A nibble-value that specifies the "encryptingMethod",
        '''	The value 0x0 specifies that no encryptingMethod is used.</param>
        ''' <param name="MemoryAddress">starting address of server memory from which data is to be retrieved</param>
        ''' <param name="MemoryAddressLength">Size in bytes of the MemoryAddress buffer (max.: 0xF)</param>
        ''' <param name="MemorySize">used by the server to compare the uncompressed memory size with
        '''	the total amount of data transferred during the TransferData service</param>
        ''' <param name="MemorySizeLength">Size in bytes of the MemorySize buffer (max.: 0xF)</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_SvcRequestUpload")> _
        Public Shared Function SvcRequestUpload( _
            ByVal CanChannel As TPUDSCANHandle, _
            ByRef MessageBuffer As TPUDSMsg, _
            ByVal CompressionMethod As Byte, _
            ByVal EncryptingMethod As Byte, _
            ByVal MemoryAddress As Byte(), _
            ByVal MemoryAddressLength As Byte, _
            ByVal MemorySize As Byte(), _
            ByVal MemorySizeLength As Byte) As TPUDSStatus
        End Function
#End Region

#Region "UDS Service: TransferData"
        '' ISO-15765-3:2004 §9.7.1 p.57 && ISO-14229-1:2006 §14.4 p.237

        ''' <summary>
        '''	The TransferData service is used by the client to transfer data either from the client
        '''	to the server (download) or from the server to the client (upload).
        ''' </summary>
        ''' <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        ''' <param name="MessageBuffer">The PUDS message (NO_POSITIVE_RESPONSE_MSG is ignored)</param>
        ''' <param name="BlockSequenceCounter">The blockSequenceCounter parameter value starts at 01 hex
        '''	with the first TransferData request that follows the RequestDownload (34 hex)
        '''	or RequestUpload (35 hex) service. Its value is incremented by 1 for each subsequent
        '''	TransferData request. At the value of FF hex, the blockSequenceCounter rolls over
        '''	and starts at 00 hex with the next TransferData request message.</param>
        ''' <param name="Buffer">buffer containing the required transfer parameters</param>
        ''' <param name="BufferLength">Size in bytes of the buffer</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_SvcTransferData")> _
        Public Shared Function SvcTransferData( _
            ByVal CanChannel As TPUDSCANHandle, _
            ByRef MessageBuffer As TPUDSMsg, _
            ByVal BlockSequenceCounter As Byte, _
            ByVal Buffer As Byte(), _
            ByVal BufferLength As UShort) As TPUDSStatus
        End Function
#End Region

#Region "UDS Service: RequestTransferExit"
        '' ISO-15765-3:2004 §9.7.1 p.57 && ISO-14229-1:2006 §14.5 p.242

        ''' <summary>
        '''	The RequestTransferExit service is used by the client to terminate a data
        '''	transfer between client and server (upload or download).
        ''' </summary>
        ''' <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        ''' <param name="MessageBuffer">The PUDS message (NO_POSITIVE_RESPONSE_MSG is ignored)</param>
        ''' <param name="Buffer">buffer containing the required transfer parameters</param>
        ''' <param name="BufferLength">Size in bytes of the buffer</param>
        ''' <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-UDS.dll", EntryPoint:="UDS_SvcRequestTransferExit")> _
        Public Shared Function SvcRequestTransferExit( _
            ByVal CanChannel As TPUDSCANHandle, _
            ByRef MessageBuffer As TPUDSMsg, _
            ByVal Buffer As Byte(), _
            ByVal BufferLength As UShort) As TPUDSStatus
        End Function
#End Region

#End Region
    End Class
#End Region
End Namespace
